#!/usr/bin/env python
import cv2
import rospy
from sensor_msgs.msg import Image
from cv_bridge import CvBridge

import numpy as np
import math
import time
import pyrealsense2 as rs


class CameraManager:
    def __init__(self):
        self.pipeline = rs.pipeline()
        self.config = rs.config()
        self.config.enable_stream(rs.stream.depth, 640, 480, rs.format.z16, 60)
        self.config.enable_stream(rs.stream.color, 640, 480, rs.format.bgr8, 60)
        
    def register(self):
        self.pipeline.start(self.config)

    # 图像服务函数，返回角度sita
    def color_srv(self):
        """
            寻找色块
        """
        frames = self.pipeline.wait_for_frames()
        color_frame = frames.get_color_frame()
        image_color = np.asanyarray(color_frame.get_data())
        color_dist = {
            'red': {'Lower': np.array([0, 60, 60]), 'Upper': np.array([6, 255, 255])},
            'blue': {'Lower': np.array([97, 224, 70]), 'Upper': np.array([124, 255, 255])},
            'green': {'Lower': np.array([35, 43, 35]), 'Upper': np.array([90, 255, 255])},
        }
        gs_frame = cv2.GaussianBlur(image_color, (5, 5), 0)  # 高斯模糊
        hsv = cv2.cvtColor(gs_frame, cv2.COLOR_BGR2HSV)  # 转化成HSV图像
        erode_hsv = cv2.erode(hsv, None, iterations=2)  # 腐蚀 粗的变细
        inRange_hsv = cv2.inRange(erode_hsv, color_dist["blue"]['Lower'],
                                  color_dist["blue"]['Upper'])  # 识别颜色
        cnts = cv2.findContours(inRange_hsv.copy(), cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE)[-2]  # 寻找轮廓
        try:
            c = max(cnts, key=cv2.contourArea)  # 寻找最大色块
        except:
            return 0
        rect = cv2.minAreaRect(c)  # 找最小的包裹色块的矩形
        box = cv2.boxPoints(rect)  # box的点
        sumx = 0.0
        sumy = 0.0
        for pi in box:
            sumx += pi[0]
            sumy += pi[1]
        center = (int(sumx / 4), int(sumy / 4))
        sita = math.atan(math.tan(34.5 / 180 * 3.1415926) * (center[0] - 320) / 320) / 3.1415926 * 180
        # print("center, sita = ", center, sita)
        # image = gs_frame
        # cv2.drawContours(image, [np.int0(box)], -1, (0, 255, 255), 2)
        # cv2.circle(image, center, 7, (0, 0, 0), -1)
        # cv2.imshow("a", image)
        # cv2.waitKey(1)
        return int(sita)

    def depth_srv(self):
        frames = self.pipeline.wait_for_frames()
        depth_frame = frames.get_depth_frame()
        depth_image = np.asanyarray(depth_frame.get_data())
        image_arr = np.array(depth_image)
        # print(image_arr.shape[0])
        divide_p = [image_arr.shape[1] // 4, image_arr.shape[1] // 2, image_arr.shape[1] * 3 // 4]
        center_part = image_arr[:int(image_arr.shape[0] * 3 // 4), int(image_arr.shape[1] * 3 // 8) : int(image_arr.shape[1] * 5 // 8)]
        center_block_cnt = np.sum(center_part < 400)
        left_part = image_arr[int(image_arr.shape[0] / 2):, 0:divide_p[1]]
        right_part = image_arr[int(image_arr.shape[0] / 2):, divide_p[1]:image_arr.shape[1]]
        # center_part_mean = center_part.mean()
        # center_per = center_block_cnt / center_part.size
        left_part_mean = left_part.mean()
        right_part_mean = right_part.mean()
        return int(center_block_cnt), int(left_part_mean), int(right_part_mean)


if __name__ == "__main__":
    import sys
    print("camera", sys.version)
    rospy.init_node("CameraManager")

    camera = CameraManager()

    rate = rospy.Rate(1000)
    rospy.spin()
    rate.sleep()
