#!/usr/bin/env python
import rospy
from std_msgs.msg import String
import time
import cmd
import logging

def SendCommand(msg):
    print(msg.data)
    pub.publish(msg)

class CommandCLI(cmd.Cmd):
    intro = 'XLAB机器人控制平台, 输入 help 或者?查看帮助。\n'
    prompt = '> 请输入指令: '
    msg = String()

    def ErrorArgs(self):
        print('参数个数错误，请检查输入')

    def checkArgs(self, arg, aim):
        '检查参数合法性; arg1:输入参数; arg2:包含符合要求的参数个数'
        args = arg.split(' ')
        argnum = 0 if arg == '' else len(args)
        invalid = False
        if(argnum not in aim):
            self.ErrorArgs()
            invalid = True
        return args, argnum, invalid

    def default(self, line):
        print('无效指令，请检查输入')

    def precmd(self, line: str) -> str:
        logger.info(f'{line}')
        return super().precmd(line)

    def do_EOF(self, line):
        '退出快捷键, 输入ctrl+D发送EOF使得控制终端退出'
        return True

    def do_scan(self, arg):
        '扫描舵机'
        args, argnum, invalid = self.checkArgs(arg, [0])
        if invalid: return 
        self.msg.data = 'manager.' + 'ServoScan()'
        SendCommand(self.msg)
    
    def do_rst(self, arg):
        '复位至初始位置'
        args, argnum, invalid = self.checkArgs(arg, [0])
        if invalid: return 
        self.msg.data = 'manager.' + 'Init()'
        SendCommand(self.msg)
    
    def do_set(self, arg):
        '设置初始位置'
        args, argnum, invalid = self.checkArgs(arg, [0])
        if invalid: return 
        self.msg.data = 'manager.' + 'SetInit()'
        SendCommand(self.msg)
    
    def do_damp(self, arg):
        '使所有舵机转变为低阻尼模式'
        args, argnum, invalid = self.checkArgs(arg, [0, 1])
        if invalid: return 
        if(argnum == 0):
            self.msg.data = 'manager.' + 'DampAll()'
        else:
            self.msg.data = 'manager.' + f'Damp({args[0]})'
        SendCommand(self.msg)
    
    def do_show(self, arg):
        '显示舵机状态'
        args = arg.split(' ')
        if(len(args) > 1): 
            self.ErrorArgs()
            return
        elif(arg == ''):
            self.msg.data = 'manager.' + f'ShowServoState()'
        else:
            self.msg.data = 'manager.' + f'ShowServoState({args[0]})'
        SendCommand(self.msg)
    
    def do_showabs(self, arg):
        '显示舵机状态'
        args = arg.split(' ')
        if(len(args) > 1): 
            self.ErrorArgs()
            return
        elif(arg == ''):
            self.msg.data = 'manager.' + f'ShowServoAbsState()'
        else:
            self.msg.data = 'manager.' + f'ShowServoAbsState({args[0]})'
        SendCommand(self.msg)
    
    def do_showf(self, arg):
        '显示文件; arg1: 子目录名; arg2: 文件名'
        args = arg.split(' ')
        if(len(args) != 2): 
            self.ErrorArgs()
            return
        self.msg.data = 'manager.' + f'ShowFile("{args[0]}", "{args[1]}")'
        SendCommand(self.msg)
    
    def do_wt(self, arg):
        '保存时间流; arg1: 时间流文件名; arg2: 时间流时长;'
        args = arg.split(' ')
        if(len(args) != 2): 
            self.ErrorArgs()
            return
        self.msg.data = 'manager.' + f'SaveServoFlow("{args[0]}", {args[1]}, "TimeFlow")'
        SendCommand(self.msg)

    def do_rt(self, arg):
        '加载时间流; arg1: 时间流文件名;'
        args = arg.split(' ')
        if(len(args) != 1): 
            self.ErrorArgs()
            return
        self.msg.data = 'manager.' + f'LoadServoFlow("{args[0]}", "TimeFlow")'
        SendCommand(self.msg)
    
    def do_ws(self, arg):
        '保存单步动作; arg1: 文件名; arg2: 单步动作序号'
        args = arg.split(' ')
        if(len(args) != 2): 
            self.ErrorArgs()
            return
        self.msg.data = 'manager.' + f'SaveServo("{args[0]}", "Step", {args[1]})'
        SendCommand(self.msg)
    
    def do_ra(self, arg):
        '读取action; arg1: action名'
        args, argnum, invalid = self.checkArgs(arg, [1])
        if invalid: return 
        self.msg.data = 'manager.' + f'ReadAction("{args[0]}")'
        SendCommand(self.msg)
    
    def do_wa(self, arg):
        '记录action; arg1: action名'
        args, argnum, invalid = self.checkArgs(arg, [1])
        if invalid: return 
        self.msg.data = 'manager.' + f'WriteAction("{args[0]}")'
        SendCommand(self.msg)

    def do_rs(self, arg):
        '加载单步动作; arg1: 文件名; arg2: 单步动作序号'
        args = arg.split(' ')
        if(len(args) != 2): 
            self.ErrorArgs()
            return
        self.msg.data = 'manager.' + f'LoadServo("{args[0]}", "Step", {args[1]})'
        SendCommand(self.msg)

    def do_exec(self, arg):
        '将传入的参数视为函数调用语句'
        args = arg.split(' ')
        self.msg.data = args[0]
        SendCommand(self.msg)

    def do_vol(self, arg):
        '获取电量'
        args = arg.split(' ')
        if(arg != ''):
            self.ErrorArgs()
            return
        self.msg.data = 'manager.' + f'GetVol()'
        SendCommand(self.msg)

    def do_exit(self, arg):
        '退出'
        print('退出')
        exit(0)

        
if __name__ == "__main__":
    # 日志系统初始化
    logger = logging.getLogger("Controller")
    logger.setLevel(logging.INFO)
    log_file = "/home/yyf/ros03/log/controller.log"
    try:
        with open(log_file, 'r') as f:
            print('打开日志成功')
    except:
        with open(log_file, 'w') as f:
            print("创建新日志")
    fh = logging.FileHandler(log_file, "a+", encoding="utf-8")
    fh.setLevel(logging.INFO)
    formatter = logging.Formatter("%(asctime)s - %(name)s - %(levelname)s - %(message)s")
    fh.setFormatter(formatter)
    logger.addHandler(fh)

    #2.初始化 ROS 节点:命名(唯一)
    rospy.init_node("Controller")
    #3.实例化 发布者 对象
    pub = rospy.Publisher("Manager", String, queue_size=1)
    #4.组织被发布的数据，并编写逻辑发布数据

    # 进入主循环
    cli = CommandCLI()
    
    cli.cmdloop()
