package main

import (
	"database/sql"
	"encoding/json"
	"log"
	"net/http"
	"strconv"
	"sync"
	"time"

	_ "github.com/mattn/go-sqlite3"
)

type UnifiedResponse struct {
	Code int         `json:"code"`
	Msg  string      `json:"msg"`
	Data interface{} `json:"data"`
}

type Comment struct {
	ID        int       `json:"id"`
	Name      string    `json:"name"`
	Content   string    `json:"content"`
	CreatedAt time.Time `json:"created_at"`
}

type AddCommentRequest struct {
	Name    string `json:"name"`
	Content string `json:"content"`
}

type GetCommentsData struct {
	Total    int       `json:"total"`
	Comments []Comment `json:"comments"`
}

var (
	comments      = make([]Comment, 0)
	mutex         = &sync.RWMutex{}
	nextCommentID = 1
)

var db *sql.DB

func initDB(dataSourceName string) (*sql.DB, error) {
	var err error
	// 打开数据库连接 (如果文件不存在会创建)
	// dataSourceName 通常是数据库文件的路径
	db, err = sql.Open("sqlite3", dataSourceName)
	if err != nil {
		log.Printf("Error opening database: %v", err)
		return nil, err
	}

	// 检查连接是否有效
	if err = db.Ping(); err != nil {
		db.Close() // 确保关闭连接
		log.Printf("Error connecting to database: %v", err)
		return nil, err
	}

	// 创建 comments 表 (如果不存在)
	createTableSQL := `
	CREATE TABLE IF NOT EXISTS comments (
		id INTEGER PRIMARY KEY AUTOINCREMENT,
		name TEXT NOT NULL,
		content TEXT NOT NULL,
		created_at DATETIME DEFAULT CURRENT_TIMESTAMP
	);`
	_, err = db.Exec(createTableSQL)
	if err != nil {
		db.Close() // 确保关闭连接
		log.Printf("Error creating comments table: %v", err)
		return nil, err
	}

	log.Printf("Database initialized successfully.")
	return db, nil
}

func sendJSONResponse(w http.ResponseWriter, statusCode int, data UnifiedResponse) {
	w.Header().Set("Content-Type", "application/json")

	w.Header().Set("Access-Control-Allow-Origin", "*")
	w.Header().Set("Access-Control-Allow-Methods", "GET, POST, OPTIONS, DELETE")
	w.Header().Set("Access-Control-Allow-Headers", "Content-Type")

	w.WriteHeader(statusCode)
	if err := json.NewEncoder(w).Encode(data); err != nil {
		log.Printf("JSON error: %v", err)
	}
}

/*
func getCommentsHandler(w http.ResponseWriter, r *http.Request) {
	if r.Method == http.MethodOptions {
		sendJSONResponse(w, http.StatusOK, UnifiedResponse{Code: 0, Msg: "OPTIONS OK"})
		return
	}
	if r.Method != http.MethodGet {
		sendJSONResponse(w, http.StatusMethodNotAllowed, UnifiedResponse{Code: 1, Msg: "invalid method"})
		return
	}

	pageStr := r.URL.Query().Get("page")
	sizeStr := r.URL.Query().Get("size")

	page, err := strconv.Atoi(pageStr)
	if err != nil || page < 1 {
		page = 1
	}

	size, err := strconv.Atoi(sizeStr)
	if err != nil || size == 0 {
		size = 10
	}

	mutex.RLock()
	defer mutex.RUnlock()

	var commentsToReturn []Comment
	total := len(comments)

	if size == -1 {
		commentsToReturn = make([]Comment, total)
		copy(commentsToReturn, comments)
	} else if total > 0 {
		start := (page - 1) * size
		end := start + size

		if start < 0 {
			start = 0
		}

		if start >= total {
			commentsToReturn = []Comment{}
		} else {
			if end > total {
				end = total
			}
			commentsToReturn = make([]Comment, end-start)
			copy(commentsToReturn, comments[start:end])
		}
	} else {
		commentsToReturn = []Comment{}
	}

	responseData := GetCommentsData{
		Total:    total,
		Comments: commentsToReturn,
	}
	sendJSONResponse(w, http.StatusOK, UnifiedResponse{Code: 0, Msg: "success", Data: responseData})
}
*/

func getCommentsHandler(database *sql.DB) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		if r.Method == http.MethodOptions {
			sendJSONResponse(w, http.StatusOK, UnifiedResponse{Code: 0, Msg: "OPTIONS OK"})
			return
		}
		if r.Method != http.MethodGet {
			sendJSONResponse(w, http.StatusMethodNotAllowed, UnifiedResponse{Code: 1, Msg: "invalid method"})
			return
		}

		pageStr := r.URL.Query().Get("page")
		sizeStr := r.URL.Query().Get("size")

		page, err := strconv.Atoi(pageStr)
		if err != nil || page < 1 {
			page = 1
		}

		size, err := strconv.Atoi(sizeStr)
		if err != nil || size == 0 {
			size = 10 // 默认每页10条
		}

		var total int
		// 查询总评论数
		row := database.QueryRow("SELECT COUNT(*) FROM comments;")
		err = row.Scan(&total)
		if err != nil {
			log.Printf("Error getting total comment count: %v", err)
			sendJSONResponse(w, http.StatusInternalServerError, UnifiedResponse{Code: 1, Msg: "failed to get total count"})
			return
		}

		var commentsToReturn []Comment
		var selectSQL string
		var args []interface{}

		if size == -1 { // 显示全部
			selectSQL = "SELECT id, name, content, created_at FROM comments ORDER BY created_at ASC;"
			// args 保持为空
		} else { // 分页
			offset := (page - 1) * size
			// 检查 offset 是否越界，虽然数据库会处理，但这里可以提前打印警告或调整
			if offset < 0 {
				offset = 0
			}
			// 使用 ? 作为占位符，LIMIT 和 OFFSET 控制分页
			selectSQL = "SELECT id, name, content, created_at FROM comments ORDER BY created_at ASC LIMIT ? OFFSET ?;"
			args = []interface{}{size, offset}
		}

		// 执行查询
		rows, err := database.Query(selectSQL, args...)
		if err != nil {
			log.Printf("Error querying comments: %v", err)
			sendJSONResponse(w, http.StatusInternalServerError, UnifiedResponse{Code: 1, Msg: "failed to retrieve comments"})
			return
		}
		defer rows.Close() // 确保关闭 rows

		// 遍历结果集
		for rows.Next() {
			var c Comment
			// 使用 Scan 将当前行的数据读入 Comment 结构体字段
			err := rows.Scan(&c.ID, &c.Name, &c.Content, &c.CreatedAt)
			if err != nil {
				log.Printf("Error scanning comment row: %v", err)
				// 扫描错误通常表示数据有问题或查询/结构体不匹配
				continue // 跳过这一条，尝试下一条
			}
			commentsToReturn = append(commentsToReturn, c)
		}

		// 检查在遍历过程中是否发生了错误
		if err = rows.Err(); err != nil {
			log.Printf("Error during comments iteration: %v", err)
			sendJSONResponse(w, http.StatusInternalServerError, UnifiedResponse{Code: 1, Msg: "error reading comments result"})
			return
		}

		responseData := GetCommentsData{
			Total:    total,
			Comments: commentsToReturn,
		}
		sendJSONResponse(w, http.StatusOK, UnifiedResponse{Code: 0, Msg: "success", Data: responseData})
	}
}

/*
	func addCommentHandler(w http.ResponseWriter, r *http.Request) {
		if r.Method == http.MethodOptions {
			sendJSONResponse(w, http.StatusOK, UnifiedResponse{Code: 0, Msg: "OPTIONS OK"})
			return
		}
		if r.Method != http.MethodPost {
			sendJSONResponse(w, http.StatusMethodNotAllowed, UnifiedResponse{Code: 1, Msg: "disapproval method"})
			return
		}

		var reqBody AddCommentRequest
		err := json.NewDecoder(r.Body).Decode(&reqBody)
		if err != nil {
			sendJSONResponse(w, http.StatusBadRequest, UnifiedResponse{Code: 1, Msg: "invalid request: " + err.Error()})
			return
		}
		defer r.Body.Close()

		if reqBody.Name == "" || reqBody.Content == "" {
			sendJSONResponse(w, http.StatusBadRequest, UnifiedResponse{Code: 1, Msg: "name and content can't be empty"})
			return
		}

		mutex.Lock()
		defer mutex.Unlock()

		newComment := Comment{
			ID:      nextCommentID,
			Name:    reqBody.Name,
			Content: reqBody.Content,
		}
		comments = append(comments, newComment)
		nextCommentID++

		log.Printf("add comment: ID=%d, Name=%s", newComment.ID, newComment.Name)
		sendJSONResponse(w, http.StatusOK, UnifiedResponse{Code: 0, Msg: "success", Data: newComment})
	}
*/
func addCommentHandler(database *sql.DB) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		if r.Method == http.MethodOptions {
			sendJSONResponse(w, http.StatusOK, UnifiedResponse{Code: 0, Msg: "OPTIONS OK"})
			return
		}
		if r.Method != http.MethodPost {
			sendJSONResponse(w, http.StatusMethodNotAllowed, UnifiedResponse{Code: 1, Msg: "disapproval method"})
			return
		}

		var reqBody AddCommentRequest
		// 限制读取大小，防止滥用
		// r.Body = http.MaxBytesReader(w, r.Body, 1048576) // 例如限制1MB
		err := json.NewDecoder(r.Body).Decode(&reqBody)
		if err != nil {
			// 检查是否是读取大小限制错误
			// if _, ok := err.(*http.MaxBytesError); ok {
			// 	sendJSONResponse(w, http.StatusBadRequest, UnifiedResponse{Code: 1, Msg: "Request body too large"})
			// 	return
			// }
			log.Printf("Error decoding request body: %v", err)
			sendJSONResponse(w, http.StatusBadRequest, UnifiedResponse{Code: 1, Msg: "invalid request format"})
			return
		}
		defer r.Body.Close() // 确保关闭请求体

		if reqBody.Name == "" || reqBody.Content == "" {
			sendJSONResponse(w, http.StatusBadRequest, UnifiedResponse{Code: 1, Msg: "name and content can't be empty"})
			return
		}

		// 插入数据到数据库
		insertSQL := "INSERT INTO comments (name, content) VALUES (?, ?);"
		// Exec 用于执行 INSERT, UPDATE, DELETE 等不返回行的语句
		result, err := database.Exec(insertSQL, reqBody.Name, reqBody.Content)
		if err != nil {
			log.Printf("Error inserting comment: %v", err)
			sendJSONResponse(w, http.StatusInternalServerError, UnifiedResponse{Code: 1, Msg: "failed to save comment"})
			return
		}

		// 获取新插入记录的ID
		id, err := result.LastInsertId()
		if err != nil {
			log.Printf("Error getting last insert ID: %v", err)
			// 即使获取ID失败，评论可能已经成功插入
			// 可以选择返回成功，或者根据业务决定是否算失败
			sendJSONResponse(w, http.StatusInternalServerError, UnifiedResponse{Code: 1, Msg: "failed to get new comment ID"})
			return
		}

		log.Printf("add comment: ID=%d, Name=%s", id, reqBody.Name)
		// 返回新评论的ID (可选，取决于前端是否需要立即知道)
		// 可以通过查询刚刚插入的数据获取完整的 Comment 结构体，包括 created_at
		// 或者只返回ID
		sendJSONResponse(w, http.StatusOK, UnifiedResponse{Code: 0, Msg: "success", Data: map[string]int{"id": int(id)}})
	}
}

/*
func deleteCommentHandler(w http.ResponseWriter, r *http.Request) {
	if r.Method == http.MethodOptions {
		sendJSONResponse(w, http.StatusOK, UnifiedResponse{Code: 0, Msg: "options ok"})
		return
	}
	if r.Method != http.MethodPost {
		sendJSONResponse(w, http.StatusMethodNotAllowed, UnifiedResponse{Code: 1, Msg: "invalid method"})
		return
	}

	idStr := r.URL.Query().Get("id")
	commentID, err := strconv.Atoi(idStr)
	if err != nil {
		sendJSONResponse(w, http.StatusBadRequest, UnifiedResponse{Code: 1, Msg: "invalid comment ID"})
		return
	}

	mutex.Lock()
	defer mutex.Unlock()

	found := false
	newCommentsList := make([]Comment, 0, len(comments))
	for _, c := range comments {
		if c.ID == commentID {
			found = true
		} else {
			newCommentsList = append(newCommentsList, c)
		}
	}

	if !found {
		sendJSONResponse(w, http.StatusNotFound, UnifiedResponse{Code: 1, Msg: "评论未找到"})
		return
	}

	comments = newCommentsList
	log.Printf("删除评论 ID: %d", commentID)
	sendJSONResponse(w, http.StatusOK, UnifiedResponse{Code: 0, Msg: "success", Data: nil}) // [cite: 40]
}
*/

func deleteCommentHandler(database *sql.DB) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		if r.Method == http.MethodOptions {
			sendJSONResponse(w, http.StatusOK, UnifiedResponse{Code: 0, Msg: "options ok"})
			return
		}
		// 前端使用POST方法删除，所以这里保持POST
		if r.Method != http.MethodPost {
			sendJSONResponse(w, http.StatusMethodNotAllowed, UnifiedResponse{Code: 1, Msg: "invalid method"})
			return
		}

		idStr := r.URL.Query().Get("id")
		commentID, err := strconv.Atoi(idStr)
		if err != nil {
			sendJSONResponse(w, http.StatusBadRequest, UnifiedResponse{Code: 1, Msg: "invalid comment ID format"})
			return
		}

		// 执行删除操作
		deleteSQL := "DELETE FROM comments WHERE id = ?;"
		result, err := database.Exec(deleteSQL, commentID)
		if err != nil {
			log.Printf("Error deleting comment ID %d: %v", commentID, err)
			sendJSONResponse(w, http.StatusInternalServerError, UnifiedResponse{Code: 1, Msg: "failed to delete comment"})
			return
		}

		// 检查影响的行数，判断是否删除了记录
		rowsAffected, err := result.RowsAffected()
		if err != nil {
			log.Printf("Error getting rows affected for delete ID %d: %v", commentID, err)
			// 即使获取行数失败，删除操作可能已经成功，或者本来就没找到，这里可以返回成功或根据业务决定
			sendJSONResponse(w, http.StatusOK, UnifiedResponse{Code: 0, Msg: "delete operation attempted, status unknown"})
			return
		}

		if rowsAffected == 0 {
			// 没有找到匹配ID的评论
			log.Printf("Delete comment ID %d not found", commentID)
			sendJSONResponse(w, http.StatusNotFound, UnifiedResponse{Code: 1, Msg: "评论未找到"})
			return
		}

		log.Printf("删除评论 ID: %d", commentID)
		sendJSONResponse(w, http.StatusOK, UnifiedResponse{Code: 0, Msg: "success", Data: nil})
	}
}

/*
func loggingMiddleware(next http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		start := time.Now()

		lrw := &loggingResponseWriter{ResponseWriter: w, statusCode: http.StatusOK}

		next.ServeHTTP(lrw, r)

		log.Printf(
			"request: %s %s, answer: %d, time_comsume: %s",
			r.Method,
			r.RequestURI,
			lrw.statusCode,
			time.Since(start),
		)
	})
}
*/

func loggingMiddleware(next http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		start := time.Now()
		lrw := &loggingResponseWriter{ResponseWriter: w, statusCode: http.StatusOK}
		next.ServeHTTP(lrw, r)
		log.Printf(
			"request: %s %s, answer: %d, time_comsume: %s",
			r.Method,
			r.RequestURI,
			lrw.statusCode,
			time.Since(start),
		)
	})
}

type loggingResponseWriter struct {
	http.ResponseWriter
	statusCode int
}

func (lrw *loggingResponseWriter) WriteHeader(code int) {
	lrw.statusCode = code
	lrw.ResponseWriter.WriteHeader(code)
}

func timeoutMiddleware(next http.Handler, duration time.Duration) http.Handler {
	return http.TimeoutHandler(next, duration, "time exceed")
}

/*
func loadInitialData() {
	mutex.Lock()
	defer mutex.Unlock()
	if len(comments) == 0 {
		comments = append(comments, Comment{ID: nextCommentID, Name: "thy", Content: "这是第一条评论qaq"})
		nextCommentID++
		comments = append(comments, Comment{ID: nextCommentID, Name: "thy", Content: "这是第一条评论qwq"})
		nextCommentID++
		comments = append(comments, Comment{ID: nextCommentID, Name: "thy", Content: "这是第一条评论awa"})
		nextCommentID++
		log.Println("initial comments loading successfully")
	}
}
*/

func loadInitialData(database *sql.DB) {
	// 检查数据库是否已经有评论
	var count int
	row := database.QueryRow("SELECT COUNT(*) FROM comments;")
	err := row.Scan(&count)
	if err != nil {
		log.Printf("Error checking initial data count: %v", err)
		return
	}
	if count == 0 {
		log.Println("No comments found, inserting initial data.")
		// 使用事务插入多条数据
		tx, err := database.Begin()
		if err != nil {
			log.Printf("Error starting transaction for initial data: %v", err)
			return
		}
		defer tx.Rollback() // 如果后续操作失败，回滚事务

		insertSQL := "INSERT INTO comments (name, content) VALUES (?, ?);"
		stmt, err := tx.Prepare(insertSQL)
		if err != nil {
			log.Printf("Error preparing statement for initial data: %v", err)
			return
		}
		defer stmt.Close()

		initialComments := []struct{ Name, Content string }{
			{"thyA", "这是第一条初始化评论！"},
			{"thyB", "这是第二条初始化评论！"},
			{"thyC", "这是第三条初始化评论！"},
		}

		for _, c := range initialComments {
			_, err := stmt.Exec(c.Name, c.Content)
			if err != nil {
				log.Printf("Error inserting initial comment (%s): %v", c.Name, err)
				return // 插入失败，回滚整个事务
			}
		}

		err = tx.Commit() // 提交事务
		if err != nil {
			log.Printf("Error committing transaction for initial data: %v", err)
			return
		}
		log.Println("Initial comments loaded successfully into database.")
	} else {
		log.Printf("Database already contains %d comments, skipping initial data load.", count)
	}
}

func main() {

	var err error
	db, err = initDB("comments.db")
	if err != nil {
		log.Fatalf("Failed to initialize database: %v", err)
	}
	defer db.Close()

	loadInitialData(db)

	mux := http.NewServeMux()
	/*
		mux.HandleFunc("/comment/get", getCommentsHandler)
		mux.HandleFunc("/comment/add", addCommentHandler)
		mux.HandleFunc("/comment/delete", deleteCommentHandler)
	*/

	mux.HandleFunc("/comment/get", getCommentsHandler(db))
	mux.HandleFunc("/comment/add", addCommentHandler(db))
	mux.HandleFunc("/comment/delete", deleteCommentHandler(db))

	loggedMux := loggingMiddleware(mux)

	port := ":8080"
	log.Printf("server is current working on %s\n", port)

	if err := http.ListenAndServe(port, loggedMux); err != nil {
		log.Fatalf("can't start server: %s\n", err.Error())
	}
}
