document.addEventListener('DOMContentLoaded', () => {
    // 修复变量名和对应的HTML ID
    const userNameInput = document.getElementById('username'); 
    const commentContentInput = document.getElementById('comment-text'); 
    const submitBtn = document.getElementById('submit-comment'); 

    const prevPageBtn = document.getElementById('prev-page');
    const nextPageBtn = document.getElementById('next-page'); 
    const currentPageSpan = document.getElementById('current-page'); 
    const pageSizeSelect = document.getElementById('pageSizeSelect'); 

    const commentsList = document.getElementById('comments-list');

    const API_HOST = 'http://localhost:8080'; 
    
    let currentPage = 1;
    let pageSize = parseInt(pageSizeSelect.value, 10);
    let totalComments = 0;

    async function fetchComments() {
        try {
            const response = await fetch(`${API_HOST}/comment/get?page=${currentPage}&size=${pageSize}`);
            if (!response.ok) {
                throw new Error(`HTTP error: ${response.status}`);
            }
            const result = await response.json();

            if (result.code === 0) { 
                renderComments(result.data.comments);
                totalComments = result.data.total;
                updatePaginationControls();
            } else {
                commentsList.innerHTML = `<p>加载评论失败: ${escapeHTML(result.msg)}</p>`; // 显示错误信息
                totalComments = 0; 
                updatePaginationControls(); 
                alert(`无法加载评论: ${result.msg}`);
            }
        } catch (error) {
            console.error('加载评论错误: ', error);
            commentsList.innerHTML = '<p>加载评论出错，请检查服务器是否运行。</p>'; 
            totalComments = 0; 
            updatePaginationControls(); 
            alert('无法加载评论，请检查后端服务器是否正常运行？');
        }
    }

    function renderComments(comments = []) {
        commentsList.innerHTML = '';

        if (!comments || comments.length === 0) {
            commentsList.innerHTML = '<p>暂无评论。</p>';
            return;
        }

        comments.forEach(comment => {
            const commentItem = document.createElement('div');
            commentItem.classList.add('comment-item');
            commentItem.dataset.id = comment.id;

            commentItem.innerHTML = `
                <span class="user-name">${escapeHTML(comment.name)}</span>
                <p class="comment-text">${escapeHTML(comment.content)}</p>
                <button class="delete-btn">删除</button>
            `;
            commentsList.appendChild(commentItem);
        });
    }

    async function addComment() {
        const userName = userNameInput.value.trim();
        const content = commentContentInput.value.trim();

        if (!userName || !content) {
            alert('用户名和评论内容不能为空');
            return;
        }

        try {
            const response = await fetch(`${API_HOST}/comment/add`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ name: userName, content: content }),
            });

            if (!response.ok) {
                throw new Error(`HTTP error: ${response.status}`);
            }
            const result = await response.json();

            if (result.code === 0) { 
                currentPage = 1; 
                fetchComments(); 
                userNameInput.value = '';
                commentContentInput.value = '';
            } else {
                alert(`添加评论失败: ${result.msg}`);
            }
        } catch (error) {
            console.error('添加评论错误', error);
            alert('添加评论出错');
        }
    }

    async function handleDelete(event) {
        if (event.target.classList.contains('delete-btn')) {
            const commentItem = event.target.closest('.comment-item');
            if (commentItem) {
                const commentId = commentItem.dataset.id;
                if (confirm(`确定要删除ID为 ${commentId} 的评论吗？`)) {
                    try {
                        const response = await fetch(`${API_HOST}/comment/delete?id=${commentId}`, {
                            method: 'POST', 
                        });
                        if (!response.ok) {
                             // 对于非2xx状态码，尝试读取后端返回的错误信息
                            const errorData = await response.json().catch(() => ({ msg: `HTTP status ${response.status}` }));
                            throw new Error(`删除失败: ${errorData.msg || `HTTP status ${response.status}`}`);
                        }
                        const result = await response.json();

                        if (result.code === 0) {
                            fetchComments(); 
                        } else {
                            alert(`删除失败: ${result.msg}`);
                        }
                    } catch (error) {
                        console.error('删除评论错误', error);
                        alert(`删除评论出错: ${error.message}`); 
                    }
                }
            }
        }
    }

    function updatePaginationControls() {
        currentPageSpan.textContent = `Page: ${currentPage}`;
        
        const maxPage = (pageSize === -1 || totalComments === 0) ? 1 : Math.ceil(totalComments / pageSize);

        prevPageBtn.disabled = currentPage <= 1; 
        nextPageBtn.disabled = currentPage >= maxPage; 

        if (pageSize === -1) { 
             prevPageBtn.disabled = true;
             nextPageBtn.disabled = true;
        }
    }

    function escapeHTML(str) {
        const div = document.createElement('div');
        div.appendChild(document.createTextNode(str));
        return div.innerHTML;
    }


    if(submitBtn) submitBtn.addEventListener('click', addComment);
    if(commentsList) commentsList.addEventListener('click', handleDelete);
    
    if(prevPageBtn) {
        prevPageBtn.addEventListener('click', () => {
            if (currentPage > 1) {
                currentPage--;
                fetchComments();
            }
        });
    }

    if(nextPageBtn) {
        nextPageBtn.addEventListener('click', () => {
            const maxPage = (pageSize === -1 || totalComments === 0) ? 1 : Math.ceil(totalComments / pageSize);
            if (currentPage < maxPage || pageSize === -1) { 
                 if (pageSize === -1 && currentPage === 1) {
                     // 如果是显示全部模式且在第一页，点击下一页不应该有反应
                     return;
                 }
                 if (pageSize !== -1 && currentPage >= maxPage) {
                     // 如果不是显示全部模式且已经是最后一页，不应该有反应
                     return;
                 }

                currentPage++;
                fetchComments();
            }
        });
    }

    if(pageSizeSelect) {
        pageSizeSelect.addEventListener('change', (event) => {
            pageSize = parseInt(event.target.value, 10);
            currentPage = 1;
            fetchComments();
        });
    }


    fetchComments();
});