'use client';
import React, { useState, useEffect, useRef } from 'react';
import useSWR from 'swr';
import useSWRMutation from 'swr/mutation';

interface ApiResponse<T> {
  message: string;
  code: number;
  data: T;
}
interface Message {
  messageId: number;
  roomId: number;
  sender: string;
  content: string;
  time: number;
}
interface Room {
  roomId: number;
  roomName: string;
  lastMessage: Message | null;
}
type RoomListRes = { rooms: Room[] };
type RoomMessageListRes = { messages: Message[] };

const API_PREFIX = "https://chatroom.zjuxlab.com";

const getFetcher = async (key: string) => {
  const res = await fetch(API_PREFIX + key, { mode: "cors" });
  if (!res.ok) {
    throw new Error('网络请求失败');
  }
  const data: ApiResponse<any> = await res.json();
  if (data.code !== 0) {
    throw new Error(data.message || '获取数据失败');
  }
  return data.data; 
};

const mutationFetcher = async (key: string, { arg }: { arg: Record<string, unknown> }) => {
    const res = await fetch(API_PREFIX + key, {
        method: 'POST', 
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(arg),
        mode: "cors",
    });
    if (!res.ok) {
        throw new Error('网络请求失败');
    }
    const data: ApiResponse<any> = await res.json();
    if (data.code !== 0) {
        throw new Error(data.message || '操作失败');
    }
    return data.data;
};

const RoomEntry = ({ room, isSelected, onClick, onDelete }: { room: Room, isSelected: boolean, onClick: () => void, onDelete: (e: React.MouseEvent) => void }) => (
  <div 
    onClick={onClick}
    className={`relative group p-3 my-1 flex justify-between items-center rounded-lg cursor-pointer transition-all duration-200 ${isSelected ? 'bg-blue-500 text-white' : 'hover:bg-gray-700'}`}
  >
    <div className="truncate">
      <p className="font-semibold">{room.roomName}</p>
      <p className={`text-xs truncate ${isSelected ? 'text-blue-200' : 'text-gray-400'}`}>
        {room.lastMessage ? room.lastMessage.content : '暂无消息'}
      </p>
    </div>
    <button 
      onClick={onDelete}
      className="absolute right-2 top-1/2 -translate-y-1/2 p-1 rounded-full bg-red-500 text-white text-xs opacity-0 group-hover:opacity-100 transition-opacity focus:opacity-100"
      aria-label="删除房间"
    >
      ✕
    </button>
  </div>
);

const MessageItem = ({ message, isOwnMessage }: { message: Message, isOwnMessage: boolean }) => {
  const avatarInitial = message.sender ? message.sender.charAt(0).toUpperCase() : '?';
  const messageDate = new Date(message.time);
  const timeString = messageDate.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });

  return (
    <div className={`flex items-start my-4 gap-3 ${isOwnMessage ? 'flex-row-reverse' : ''}`}>
      <div className="w-10 h-10 rounded-full bg-gray-600 flex-shrink-0 flex items-center justify-center font-bold text-white">
        {avatarInitial}
      </div>
      <div className={`flex flex-col ${isOwnMessage ? 'items-end' : 'items-start'}`}>
        <div className="flex items-center gap-2">
           {!isOwnMessage && <span className="font-semibold text-sm text-gray-300">{message.sender}</span>}
           <span className="text-xs text-gray-500">{timeString}</span>
        </div>
        <div className={`mt-1 p-3 rounded-lg max-w-xs md:max-w-md break-words ${isOwnMessage ? 'bg-blue-600 text-white' : 'bg-gray-700 text-gray-200'}`}>
          {message.content}
        </div>
      </div>
    </div>
  );
};

const ChatRoomPage = ({ nickname, onLogout }: { nickname: string, onLogout: () => void }) => {
  const [selectedRoomId, setSelectedRoomId] = useState<number | null>(null);
  const [newMessage, setNewMessage] = useState('');
  const [newRoomName, setNewRoomName] = useState('');
  const messagesEndRef = useRef<HTMLDivElement>(null);

  const { data: roomsData, error: roomsError, isLoading: roomsLoading } = useSWR<RoomListRes>('/api/room/list', getFetcher, { refreshInterval: 1000 });

  const { data: messagesData, error: messagesError, isLoading: messagesLoading } = useSWR<RoomMessageListRes>(
    () => selectedRoomId !== null ? `/api/room/message/list?roomId=${selectedRoomId}` : null,
    getFetcher,
    { refreshInterval: 1000 }
  );

  const { trigger: addRoomTrigger } = useSWRMutation('/api/room/add', mutationFetcher);
  const { trigger: deleteRoomTrigger } = useSWRMutation('/api/room/delete', mutationFetcher);
  const { trigger: addMessageTrigger } = useSWRMutation('/api/message/add', mutationFetcher);

  useEffect(() => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  }, [messagesData]);
  
  useEffect(() => {
    const lastRoomId = localStorage.getItem('lastSelectedRoomId');
    if (lastRoomId) {
      setSelectedRoomId(Number(lastRoomId));
    }
  }, []);

  const handleRoomSelect = (roomId: number) => {
    setSelectedRoomId(roomId);
    localStorage.setItem('lastSelectedRoomId', String(roomId));
  };

  const handleAddRoom = async () => {
    if (!newRoomName.trim()) return;
    try {
      await addRoomTrigger({ user: nickname, roomName: newRoomName });
      setNewRoomName('');
    } catch (e) {
      console.error(e);
      alert((e as Error).message);
    }
  };

  const handleDeleteRoom = async (e: React.MouseEvent, roomId: number) => {
    e.stopPropagation(); 
    if (confirm('确定要删除这个房间吗？')) {
      try {
        await deleteRoomTrigger({ user: nickname, roomId });
        if (selectedRoomId === roomId) {
          setSelectedRoomId(null);
        }
      } catch (e) {
        console.error(e);
        alert((e as Error).message);
      }
    }
  };
  
  const handleSendMessage = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!newMessage.trim() || selectedRoomId === null) return;
    try {
      setNewMessage(''); 
      await addMessageTrigger({
        roomId: selectedRoomId,
        sender: nickname,
        content: newMessage,
      });
    } catch (error) {
        console.error('发送失败:', error);
        alert((error as Error).message);
        setNewMessage(newMessage); 
    }
  };
  
  const currentRoomName = roomsData?.rooms.find(r => r.roomId === selectedRoomId)?.roomName || '聊天室';
  const rooms = roomsData?.rooms;
  const messages = messagesData?.messages;

  return (
    <div className="flex h-screen bg-gray-900 text-white font-sans">
      <aside className="w-1/4 min-w-[280px] bg-gray-800 flex flex-col p-4 border-r border-gray-700">
        <header className="mb-4">
          <h1 className="text-2xl font-bold">聊天室</h1>
          <p className="text-sm text-gray-400">你好, {nickname} <button onClick={onLogout} className="ml-2 text-red-400 hover:underline text-xs">[退出]</button></p>
        </header>
        <div className="flex gap-2 mb-4">
          <input
            type="text"
            value={newRoomName}
            onChange={(e) => setNewRoomName(e.target.value)}
            onKeyDown={(e) => e.key === 'Enter' && handleAddRoom()}
            placeholder="新房间名称"
            className="flex-grow p-2 bg-gray-700 rounded-lg border-2 border-transparent focus:outline-none focus:border-blue-500 transition"
          />
          <button onClick={handleAddRoom} className="p-2 bg-blue-600 rounded-lg hover:bg-blue-700 transition font-semibold">+</button>
        </div>
        <div className="flex-1 overflow-y-auto pr-1">
          {roomsLoading && <p>加载房间中...</p>}
          {roomsError && <p className="text-red-400">加载房间失败: {roomsError.message}</p>}
          {rooms && rooms.map(room => (
            <RoomEntry
              key={room.roomId}
              room={room}
              isSelected={room.roomId === selectedRoomId}
              onClick={() => handleRoomSelect(room.roomId)}
              onDelete={(e) => handleDeleteRoom(e, room.roomId)}
            />
          ))}
        </div>
      </aside>

      <main className="flex-1 flex flex-col">
        {selectedRoomId === null ? (
          <div className="flex-1 flex items-center justify-center">
            <p className="text-gray-500 text-lg">请从左侧选择一个房间开始聊天</p>
          </div>
        ) : (
          <>
            <header className="p-4 border-b border-gray-700">
              <h2 className="text-xl font-semibold">{currentRoomName}</h2>
            </header>
            <div className="flex-1 p-4 overflow-y-auto">
              {messagesLoading && <p>加载消息中...</p>}
              {messagesError && <p className="text-red-400">加载消息失败: {messagesError.message}</p>}
              {messages && messages.map(msg => (
                <MessageItem key={msg.messageId} message={msg} isOwnMessage={msg.sender === nickname} />
              ))}
              <div ref={messagesEndRef} />
            </div>
            <div className="p-4 border-t border-gray-700">
              <form onSubmit={handleSendMessage} className="flex gap-4">
                <input
                  type="text"
                  value={newMessage}
                  onChange={(e) => setNewMessage(e.target.value)}
                  placeholder="输入消息..."
                  className="flex-grow p-3 bg-gray-800 rounded-lg border-2 border-transparent focus:outline-none focus:border-blue-500 transition"
                  autoFocus
                />
                <button type="submit" className="px-6 py-3 bg-blue-600 rounded-lg font-semibold hover:bg-blue-700 transition disabled:bg-gray-600" disabled={!newMessage.trim()}>
                  发送
                </button>
              </form>
            </div>
          </>
        )}
      </main>
    </div>
  );
};

const SetNamePage = ({ onNameSubmit }: { onNameSubmit: (name: string) => void }) => {
  const [name, setName] = useState('');

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (name.trim()) {
      onNameSubmit(name.trim());
    }
  };

  return (
    <div className="flex items-center justify-center h-screen bg-gray-900 text-white">
      <div className="w-full max-w-sm p-8 bg-gray-800 rounded-2xl shadow-lg border border-gray-700">
        <h1 className="text-3xl font-bold text-center mb-2">欢迎来到聊天室</h1>
        <p className="text-center text-gray-400 mb-8">请设置你的聊天昵称</p>
        <form onSubmit={handleSubmit} className="flex flex-col gap-6">
          <input
            type="text"
            value={name}
            onChange={(e) => setName(e.target.value)}
            placeholder="输入你的昵称"
            className="p-4 text-lg bg-gray-700 rounded-lg border-2 border-transparent focus:outline-none focus:border-blue-500 transition"
            autoFocus
          />
          <button type="submit" className="p-4 text-lg font-bold bg-blue-600 rounded-lg hover:bg-blue-700 transition disabled:opacity-50" disabled={!name.trim()}>
            进入聊天室
          </button>
        </form>
      </div>
    </div>
  );
};


export default function Home() {
  const [nickname, setNickname] = useState<string | null>(null);

  useEffect(() => {
    const storedName = localStorage.getItem('chat-nickname');
    if (storedName) {
      setNickname(storedName);
    }
  }, []);

  const handleNameSubmit = (name: string) => {
    localStorage.setItem('chat-nickname', name);
    setNickname(name);
  };
  
  const handleLogout = () => {
    localStorage.removeItem('chat-nickname');
    localStorage.removeItem('lastSelectedRoomId');
    setNickname(null);
  }

  if (!nickname) {
    return <SetNamePage onNameSubmit={handleNameSubmit} />;
  }

  return <ChatRoomPage nickname={nickname} onLogout={handleLogout}/>;
}


