import { NextRequest, NextResponse } from 'next/server';
import prisma from '@/lib/prisma';
import { getUserFromRequest } from '@/lib/auth';

export async function GET(request: NextRequest) {
    try {
        const user = await getUserFromRequest(request);
        if (!user) {
            return NextResponse.json({ message: 'Unauthorized' }, { status: 401 });
        }

        const { searchParams } = new URL(request.url);
        const roomId = searchParams.get('roomId');

        if (!roomId) {
            return NextResponse.json({ message: 'Room ID 是必需的' }, { status: 400 });
        }

        const messages = await prisma.message.findMany({
            where: {
                roomId: parseInt(roomId),
            },
            include: {
                sender: true,
            },
            orderBy: {
                createdAt: 'asc',
            },
        });
        
        const formattedMessages = messages.map(msg => ({
            messageId: msg.messageId,
            roomId: msg.roomId,
            sender: msg.sender.username,
            content: msg.content,
            time: msg.createdAt.getTime(),
        }));

        return NextResponse.json({ message: "获取成功", data: { messages: formattedMessages } });

    } catch (error) {
        console.error('获取消息列表时出错:', error);
        return NextResponse.json({ message: '服务器内部错误' }, { status: 500 });
    }
}
