import { NextRequest, NextResponse } from 'next/server';
import { getUserFromRequest } from '@/lib/auth';
import prisma from '@/lib/prisma';

export async function POST(request: NextRequest) {
    try {
        const user = await getUserFromRequest(request);
        if (!user) {
            return NextResponse.json({ message: 'Unauthorized' }, { status: 401 });
        }

        const { roomId, content } = await request.json();

        if (!roomId || !content || typeof roomId !== 'number' || typeof content !== 'string' || content.trim() === '') {
            return NextResponse.json({ message: '有效的 Room ID 和消息内容是必需的' }, { status: 400 });
        }

        const roomExists = await prisma.room.findUnique({
            where: { roomId },
        });

        if (!roomExists) {
            return NextResponse.json({ message: '房间不存在' }, { status: 404 });
        }

        const newMessage = await prisma.message.create({
            data: {
                content: content.trim(),
                roomId: roomId,
                senderId: user.userId,
            },
            include: {
                sender: true
            }
        });
        
            const formattedMessage = {
                messageId: newMessage.messageId,
                roomId: newMessage.roomId,
                sender: newMessage.sender.username,
                content: newMessage.content,
                createdAt: newMessage.createdAt.getTime(),
            };


        return NextResponse.json({ message: '消息发送成功', data: formattedMessage });

    } catch (error) {
        console.error('发送消息时出错:', error);
        return NextResponse.json({ message: '服务器内部错误' }, { status: 500 });
    }
}
