import { NextRequest, NextResponse } from 'next/server';
import prisma from '@/lib/prisma';
import bcrypt from 'bcryptjs';
import jwt from 'jsonwebtoken';

export async function POST(request: NextRequest) {
    try {
        const { username, password } = await request.json();

        if (!username || !password) {
            return NextResponse.json({ message: '用户名和密码是必需的' }, { status: 400 });
        }
        if (password.length < 6) {
            return NextResponse.json({ message: '密码长度至少需要6位' }, { status: 400 });
        }

        const existingUser = await prisma.user.findUnique({
            where: { username },
        });

        if (existingUser) {
            return NextResponse.json({ message: '该用户名已被注册' }, { status: 409 });
        }

        const hashedPassword = bcrypt.hashSync(password, 10);

        const newUser = await prisma.user.create({
            data: {
                username: username,
                password: hashedPassword,
            },
        });

        const token = jwt.sign({ userId: newUser.userId }, process.env.JWT_SECRET!, {
            expiresIn: '1h',
        });

        const response = NextResponse.json({
            message: '用户注册成功',
            data: { id: newUser.userId, username: newUser.username },
        });

        response.cookies.set('auth_token', token, {
            httpOnly: true,
            secure: process.env.NODE_ENV !== 'development',
            sameSite: 'strict',
            maxAge: 3600,
            path: '/',
        });

        return response;


    } catch (error) {
        console.error('注册时出错:', error);
        return NextResponse.json({ message: '服务器内部错误' }, { status: 500 });
    }
}
