import { NextRequest, NextResponse } from 'next/server';
import prisma from '@/lib/prisma';
import bcrypt from 'bcryptjs';
import jwt from 'jsonwebtoken';

export async function POST(request: NextRequest) {
    try {
        const { username, password } = await request.json();

        if (!username || !password) {
            return NextResponse.json({ message: '用户名和密码是必需的' }, { status: 400 });
        }

        const user = await prisma.user.findUnique({
            where: { username },
        });

        if (!user || !bcrypt.compareSync(password, user.password)) {
            return NextResponse.json({ message: '用户名或密码无效' }, { status: 401 });
        }

        const token = jwt.sign({ userId: user.userId }, process.env.JWT_SECRET!, {
            expiresIn: '1h',
        });

        const response = NextResponse.json({
            message: '登录成功',
            data: { id: user.userId, username: user.username },
        });

        response.cookies.set('auth_token', token, {
            httpOnly: true,
            secure: process.env.NODE_ENV !== 'development',
            sameSite: 'strict',
            maxAge: 3600,
            path: '/',
        });

        return response;

    } catch (error) {
        console.error('登录时出错:', error);
        return NextResponse.json({ message: '服务器内部错误' }, { status: 500 });
    }
}




