import { NextRequest, NextResponse } from 'next/server';
import prisma from '@/lib/prisma';
import { getUserFromRequest } from '@/lib/auth';

export async function GET(request: NextRequest) {
    try {
        const user = await getUserFromRequest(request);
        if (!user) {
            return NextResponse.json({ message: 'Unauthorized' }, { status: 401 });
        }

        const rooms = await prisma.room.findMany({
            include: {
                messages: {
                    orderBy: {
                        createdAt: 'desc',
                    },
                    take: 1,
                    include: {
                        sender: true
                    }
                },
            },
            orderBy: {
                createdAt: 'desc'
            }
        });

        const formattedRooms = rooms.map(room => {
            const lastMessage = room.messages[0];
            return {
                roomId: room.roomId,
                roomName: room.roomName,
                lastMessage: lastMessage ? {
                    messageId: lastMessage.messageId,
                    roomId: lastMessage.roomId,
                    sender: lastMessage.sender.username,
                    content: lastMessage.content,
                    time: lastMessage.createdAt.getTime(),
                } : null,
            };
        });

        return NextResponse.json({ message: "获取成功", data: { rooms: formattedRooms } });

    } catch (error) {
        console.error('获取房间列表时出错:', error);
        return NextResponse.json({ message: '服务器内部错误' }, { status: 500 });
    }
}
