package controller

import (
	"bytes"
	"fmt"
	"go-svc-tpl/api/dto"
	"go-svc-tpl/internal/dao"
	"go-svc-tpl/internal/dao/model"
	"net/http"
	"path"
	"strings"

	"github.com/dchest/captcha"
	"github.com/gin-gonic/gin"
	"github.com/sirupsen/logrus"
	"gorm.io/gorm"
)

// >>>>>>>>>>>>>>>>>> Interface  >>>>>>>>>>>>>>>>>>
type IServerController interface {
	Link(*gin.Context, *dto.ServerLinkReq) error
	Veri(*gin.Context, *dto.ServerVeriReq) error
	CaptchaPic(*gin.Context) error
}

// >>>>>>>>>>>>>>>>>> Controller >>>>>>>>>>>>>>>>>
var _ IServerController = (*ServerController)(nil)

var NewServerController = func() *ServerController {
	return &ServerController{}
}

type ServerController struct {
}

// 重定向到相应链接
// 当出现问题时，会按标准 response 返回，由前端显示相应页面
func (c *ServerController) Link(ctx *gin.Context, req *dto.ServerLinkReq) error {
	// 在数据库中查找对应的长链接
	var link model.Link

	if err := dao.DB(ctx).Where(&model.Link{Short: req.Short}).First(&link).Error; err != nil {
		if err == gorm.ErrRecordNotFound {
			// 如果找不到对应的长链接，返回 短连接不存在(6) 错误
			ctx.JSON(dto.ErrNoShortLink, gin.H{
				"error": "Link not found",
			})
		} else {
			// 如果发生其他错误，返回500错误
			ctx.JSON(dto.InternalError, gin.H{
				"error": "Internal server error",
			})
		}
		return err
	}
	// 如果找到了对应的长链接，进行重定向
	ctx.Redirect(http.StatusMovedPermanently, link.Origin)
	return nil
}

// 返回验证码的图片
func (c *ServerController) Veri(ctx *gin.Context, req *dto.ServerVeriReq) error {
	// 获取验证码ID
	id := req.Target
	//dto.ResponseSuccess(ctx, fmt.Sprintf("captcha id: %s", id))

	// // 判断验证码ID是否存在
	// if !captcha.Reload(id) {
	// 	return errors.New("invalid captcha id")
	// }

	// 将验证码图片写入响应体
	err := captcha.WriteImage(ctx.Writer, id, 90, 50)
	// 返回错误信息或nil
	return err
}

// 获取验证码图片
func (c *ServerController) CaptchaPic(ctx *gin.Context) error {
	imgName := ctx.Param("target")
	logrus.Info("GetCaptchaPng :" + imgName)

	ctx.Header("Access-Control-Allow-Origin", "http://localhost:8000, http://localhost:8080")
	ctx.Header("Access-Control-Allow-Credentials", "true")
	ctx.Header("Access-Control-Allow-Headers", "image/png, Content-Type, Content-Length, Accept-Encoding, X-CSRF-Token, Authorization, accept, origin, Cache-Control, X-Requested-With")
	ServeHTTP(ctx.Writer, ctx.Request, imgName)

	return nil
}

// 处理验证码图片请求
func ServeHTTP(w http.ResponseWriter, r *http.Request, img string) {
	dir, _ := path.Split(r.URL.Path)
	id := img[:len(img)-4]
	fmt.Println("id :" + id)
	if id == "" {
		http.NotFound(w, r)
		return
	}
	fmt.Println("reload :" + r.FormValue("reload"))
	if r.FormValue("reload") != "" {
		captcha.Reload(id)
	}
	lang := strings.ToLower(r.FormValue("lang"))
	download := path.Base(dir) == "download"
	if Serve(w, r, id, lang, download, 90, 50) == captcha.ErrNotFound {
		http.NotFound(w, r)
		return
	}
}

// 编辑响应头,显示验证码图片
func Serve(w http.ResponseWriter, r *http.Request, id, lang string, download bool, width, height int) error {
	w.Header().Set("Cache-Control", "no-cache,no-store,must-revalidate")
	w.Header().Set("Pragma", "no-cache")
	w.Header().Set("Expires", "0")
	w.Header().Set("Content-Type", "image/png")
	var content bytes.Buffer
	captcha.WriteImage(&content, id, width, height)
	if download {
		w.Header().Set("Content-Type", "application/octet-stream")
	}
	w.Header().Set("Access-Control-Allow-Origin", "Content-Type, image/png")
	//显示验证码图片
	w.Write(content.Bytes())

	return nil
}
