package controller

import (
	"go-svc-tpl/api/dto"
	"go-svc-tpl/internal/dao"
	"go-svc-tpl/internal/dao/model"
	"net/http"
	"regexp"

	"go-svc-tpl/utils/stacktrace"

	"bytes"
	"path"
	"strings"
	"time"

	"github.com/dchest/captcha"
	"github.com/gin-gonic/gin"
	"github.com/google/uuid"
	"golang.org/x/crypto/bcrypt"
	"gorm.io/gorm"
)

var ID uint
var sessionID string

type IUserController interface { //定义tag user 下的操作
	Register(*gin.Context, *dto.UserRegisterReq) error
	GetVeri(*gin.Context) (*dto.GetVeriResp, error)
	Login(*gin.Context, *dto.UserLoginReq) error
	GetInfo(*gin.Context) (*dto.GetUserInfoResp, error)
	ModifyInfo(*gin.Context, *dto.UserModifyInfoReq) error
	ModifyPwd(*gin.Context, *dto.UserModifyPwdReq) error
	Logout(*gin.Context) error
}

var _ IUserController = (*UserController)(nil)

var NewUserController = func() *UserController {
	return &UserController{}
}

type UserController struct {
}

// Register
func (c *UserController) Register(ctx *gin.Context, req *dto.UserRegisterReq) error {
	userID := ctx.GetUint(model.USER_ID_KEY)
	//验证邮箱格式
	emailPattern := `^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,4}$`
	_, err := regexp.MatchString(emailPattern, req.Email)
	if err != nil {
		return stacktrace.PropagateWithCode(err, dto.BadRequest, "Bad request.")
	}
	//密码Hash处理
	hashedBytes, err := bcrypt.GenerateFromPassword([]byte(req.Password), bcrypt.DefaultCost)
	if err != nil {
		return stacktrace.PropagateWithCode(err, dto.InternalError, "InternalError.")
	}
	newUser := &model.User{
		ID:       userID,
		Email:    req.Email,
		Name:     req.Name,
		Password: string(hashedBytes),
	}
	err = dao.DB(ctx).Model(&model.User{ID: userID}).
		Create(newUser).Error
	if err != nil {
		return stacktrace.PropagateWithCode(err, dto.InternalError, "Internal Error.")
	}
	//验证邮箱唯一性
	var user model.User
	err = dao.DB(ctx).Model(&model.User{ID: userID}).Where(&model.User{Email: newUser.Email}).
		First(&user).Error
	if err != nil {
		if err == gorm.ErrRecordNotFound {
			return stacktrace.PropagateWithCode(err, dto.ErrEmailExist, "ErrEmailExist.")
		}
		return stacktrace.PropagateWithCode(err, dto.InternalError, "Internal Error.")
	}
	return nil
}

// GetVeri
func (c *UserController) GetVeri(ctx *gin.Context) (*dto.GetVeriResp, error) {
	length := captcha.DefaultLen
	captchaID := captcha.NewLen(length)
	captchaURL := "/captcha/" + captchaID + ".png"
	resp := &dto.GetVeriResp{
		CAPTCHAID:  captchaID,
		CAPTCHAURL: captchaURL,
	}
	ServeHTTP(ctx.Writer, ctx.Request)
	return resp, nil
}

func ServeHTTP(w http.ResponseWriter, r *http.Request) {
	dir, file := path.Split(r.URL.Path)
	ext := path.Ext(file)
	id := file[:len(file)-len(ext)]
	if ext == "" || id == "" {
		http.NotFound(w, r)
		return
	}
	lang := strings.ToLower(r.FormValue("lang"))
	download := path.Base(dir) == "download"
	if Serve(w, r, id, ext, lang, download, captcha.StdWidth, captcha.StdHeight) == captcha.ErrNotFound {
		http.NotFound(w, r)
	}
}

func Serve(w http.ResponseWriter, r *http.Request, id, ext, lang string, download bool, width, height int) error {
	w.Header().Set("Cache-Control", "no-cache, no-store, must-revalidate")
	w.Header().Set("Pragma", "no-cache")
	w.Header().Set("Expires", "0")

	var content bytes.Buffer
	switch ext {
	case ".png":
		w.Header().Set("Content-Type", "image/png")
		captcha.WriteImage(&content, id, width, height)
	default:
		return captcha.ErrNotFound
	}

	if download {
		w.Header().Set("Content-Type", "application/octet-stream")
	}
	http.ServeContent(w, r, id+ext, time.Time{}, bytes.NewReader(content.Bytes()))
	return nil
}

// Login
func (c *UserController) Login(ctx *gin.Context, req *dto.UserLoginReq) error {
	//验证码
	captchaId := ctx.Param("captcha_id")
	value := ctx.Param("captcha_value")
	if captchaId == "" || value == "" {
		return stacktrace.PropagateWithCode(nil, dto.BadRequest, "BadRequest.")
	}
	if !captcha.VerifyString(captchaId, value) {
		return stacktrace.PropagateWithCode(nil, dto.ErrCaptcha, "ErrCaptcha.")
	}
	//找对应用户
	var user model.User
	err := dao.DB(ctx).Model(&model.User{Email: req.Email}).Where(&model.User{Email: req.Email}).First(&user).Error
	if err != nil {
		return stacktrace.PropagateWithCode(nil, dto.ErrUserNotFound, "ErrUserNotFound.")
	}
	//密码
	newUser := &model.User{
		ID:       user.ID,
		Email:    req.Email,
		Password: req.Password,
	}
	err = bcrypt.CompareHashAndPassword([]byte(user.Password), []byte(newUser.Password))
	if err != nil {
		return stacktrace.PropagateWithCode(nil, dto.ErrPassword, "ErrPassword.")
	}
	//设置cookie
	//ctx.SetCookie("id", strconv.FormatUint(uint64(userID), 10), 3600, "/", "localhost", false, true)
	ID = user.ID
	sessionID = uuid.New().String()
	ctx.SetCookie("session_id", sessionID, 86400, "/", "localhost", false, true)
	return nil
}

// Logout
func (c *UserController) Logout(ctx *gin.Context) error {
	ctx.SetCookie("id", "", -1, "/", "localhost", false, true)
	return nil
}

// GetInfo
func (c *UserController) GetInfo(ctx *gin.Context) (*dto.GetUserInfoResp, error) {
	userID := ctx.GetUint(model.USER_ID_KEY)
	var user model.User
	err := dao.DB(ctx).Model(&model.User{ID: userID}).Where(&model.User{ID: userID}).First(&user).Error
	if err != nil {
		return nil, stacktrace.PropagateWithCode(err, dto.ErrUserNotFound, "ErrUserNotFound.")
	}
	return &dto.GetUserInfoResp{
		ID:    user.ID,
		Name:  user.Name,
		Email: user.Email,
	}, nil
}

// ModifyInfo
func (c *UserController) ModifyInfo(ctx *gin.Context, req *dto.UserModifyInfoReq) error {
	userID := ctx.GetUint(model.USER_ID_KEY)
	//验证邮箱格式
	emailPattern := `^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,4}$`
	_, err := regexp.MatchString(emailPattern, req.Email)
	if err != nil {
		return stacktrace.PropagateWithCode(err, dto.ErrShortLinkExist, "ErrShortLinkExist.")
	}
	newUser := &model.User{
		ID:    userID,
		Email: req.Email,
		Name:  req.Name,
	}
	err = dao.DB(ctx).Model(&model.User{ID: userID}).Updates(&newUser).Error
	if err != nil {
		return stacktrace.PropagateWithCode(err, dto.InternalError, "Internal Error.")
	}
	return nil
}

// ModifyPwd
func (c *UserController) ModifyPwd(ctx *gin.Context, req *dto.UserModifyPwdReq) error {
	userID := ctx.GetUint(model.USER_ID_KEY)
	var user model.User
	err := dao.DB(ctx).Model(&model.User{ID: userID}).Where(&model.User{ID: userID}).
		First(&user).Error
	if err != nil {
		return stacktrace.PropagateWithCode(err, dto.ErrUserNotFound, "ErrUserNotFound.")
	}
	if user.Password != req.OldPwd {
		return stacktrace.PropagateWithCode(err, dto.ErrPassword, "ErrPassword.")
	}
	newUser := &model.User{
		Password: req.NewPwd,
	}
	err = dao.DB(ctx).Model(&model.User{ID: userID}).Updates(&newUser).Error
	if err != nil {
		return stacktrace.PropagateWithCode(err, dto.InternalError, "Internal Error.")
	}
	return nil
}
