package controller

import (
	"go-svc-tpl/api/dto"

	"context"
	"go-svc-tpl/internal/dao"
	"go-svc-tpl/internal/dao/model"
	"go-svc-tpl/utils/stacktrace"
	"net/http/httptest"
	"testing"
	"time"

	"github.com/gin-gonic/gin"
	"github.com/stretchr/testify/assert"
	"gopkg.in/guregu/null.v4"
)

func setup() {

	db := testInitDB()
	var testDB = func(ctx context.Context) *dao.DBMS {
		return &dao.DBMS{db.WithContext(ctx)}

	}
	dao.DB = testDB
	gin.SetMode(gin.TestMode)
}

func TestMain(m *testing.M) {
	setup()
	m.Run()
	teardown()
}

func teardown() {
	dao.InitDB()
	gin.SetMode(gin.ReleaseMode)
}

// TestLinkController_Create函数用于测试Create函数
func TestLinkController_Create(t *testing.T) {

	var tests = []struct {
		userID    uint
		input     dto.LinkCreateReq
		errorCode stacktrace.ErrorCode
	}{
		// // 测试用例1：正常创建短链接，不指定short字段，使用默认生成的short值
		// {
		// 	userID: 1,
		// 	input: dto.LinkCreateReq{
		// 		Short:     "",
		// 		Comment:   "test link 1",
		// 		Origin:    "https://www.google.com",
		// 		StartTime: null.TimeFrom(time.Now()),
		// 		EndTime:   null.TimeFrom(time.Now().Add(24 * time.Hour)),
		// 	},
		// 	errorCode: 0,
		// },
		// 测试用例2：正常创建短链接，指定short字段，使用自定义的short值
		{
			userID: 2,
			input: dto.LinkCreateReq{
				Short:     "bing",
				Comment:   "test link 2",
				Origin:    "https://www.bing.com",
				StartTime: null.NewTime(time.Date(2023, 7, 30, 12, 0, 12, 0, time.Local), true),
				EndTime:   null.NewTime(time.Date(2023, 8, 1, 12, 0, 12, 0, time.Local), true),
			},
			errorCode: dto.ErrShortLinkExist,
		},
		// // 测试用例3：创建短链接失败，因为short字段已经存在
		// {
		// 	userID: 3,
		// 	input: dto.LinkCreateReq{
		// 		Short:     "bing",
		// 		Comment:   "test link 3",
		// 		Origin:    "https://www.yahoo.com",
		// 		StartTime: null.TimeFrom(time.Now()),
		// 		EndTime:   null.TimeFrom(time.Now().Add(24 * time.Hour)),
		// 	},
		// 	errorCode: dto.ErrShortLinkExist,
		// },
	}

	testLinkCtl := new(LinkController)
	for _, test := range tests {
		w := httptest.NewRecorder()
		ctx, _ := gin.CreateTestContext(w)
		ctx.Set(model.USER_ID_KEY, test.userID)
		_, err := testLinkCtl.Create(ctx, &test.input)

		if test.errorCode != 0 {
			assert.Equal(t, test.errorCode, stacktrace.GetCode(err))
			continue
		}
		assert.NoError(t, err)
	}

}

func TestLinkController_Delete(t *testing.T) {

	var tests = []struct {
		userID    uint
		input     dto.LinkDeleteReq
		errorCode stacktrace.ErrorCode
	}{
		// 测试用例1：正常创建短链接，指定short字段，使用自定义的short值
		{
			userID: 2,
			input: dto.LinkDeleteReq{
				Short: "bing",
			},
			errorCode: dto.ErrShortLinkExist,
		},
	}

	testLinkCtl := new(LinkController)
	for _, test := range tests {
		w := httptest.NewRecorder()
		ctx, _ := gin.CreateTestContext(w)
		ctx.Set(model.USER_ID_KEY, test.userID)
		err := testLinkCtl.Delete(ctx, &test.input)

		if test.errorCode != 0 {
			assert.Equal(t, test.errorCode, stacktrace.GetCode(err))
			continue
		}
		assert.NoError(t, err)
	}

}

func TestGetinfo(t *testing.T) {
	var tests = []struct {
		userID    uint
		input     dto.GetLinkInfoReq
		output    dto.GetLinkInfoResp
		errorCode stacktrace.ErrorCode
	}{
		{
			userID: 2,
			input: dto.GetLinkInfoReq{
				Short: "bing",
			},
			output: dto.GetLinkInfoResp{
				Active:    null.BoolFrom(true),
				Comment:   "test link 2",
				EndTime:   null.NewTime(time.Date(2023, 8, 1, 12, 0, 12, 0, time.Local), true),
				Origin:    "https://www.bing.com",
				Short:     "short1", //注意此处和bing不一样
				StartTime: null.NewTime(time.Date(2023, 7, 30, 12, 0, 12, 0, time.Local), true),
			},
			errorCode: 0,
		},
	}
	controller := &LinkController{}
	for _, test := range tests {
		w := httptest.NewRecorder()
		ctx, _ := gin.CreateTestContext(w)
		ctx.Set(model.USER_ID_KEY, test.userID)
		resp, err := controller.GetInfo(ctx, &test.input)

		assert.Equal(t, test.output.Active, resp.Active)
		assert.Equal(t, test.output.Comment, resp.Comment)
		assert.Equal(t, test.output.EndTime, resp.EndTime)
		assert.Equal(t, test.output.Origin, resp.Origin)
		assert.Equal(t, test.output.Short, resp.Short)
		assert.Equal(t, test.output.StartTime, resp.StartTime)
		if test.errorCode != 0 {
			assert.Equal(t, test.errorCode, stacktrace.GetCode(err))

			continue
		}
		assert.NoError(t, err)
	}
}
