package controller

import (
	"go-svc-tpl/api/dto"

	"context"
	"go-svc-tpl/internal/dao"
	"go-svc-tpl/internal/dao/model"
	"go-svc-tpl/utils/stacktrace"
	"net/http/httptest"
	"testing"
	"time"

	"github.com/gin-gonic/gin"
	"github.com/stretchr/testify/assert"
	"gopkg.in/guregu/null.v4"
)

func setup() {

	db := testInitDB()
	var testDB = func(ctx context.Context) *dao.DBMS {
		return &dao.DBMS{db.WithContext(ctx)}

	}
	dao.DB = testDB
	gin.SetMode(gin.TestMode)
}

func TestMain(m *testing.M) {
	setup()
	m.Run()
	teardown()
}

func teardown() {
	dao.InitDB()
	gin.SetMode(gin.ReleaseMode)
}

// TestLinkController_Create函数用于测试Create函数
func TestCreate(t *testing.T) {

	var tests = []struct {
		userID    uint
		input     dto.LinkCreateReq
		errorCode stacktrace.ErrorCode
	}{
		// // 测试用例1：正常创建短链接，不指定short字段，使用默认生成的short值
		// {
		// 	userID: 1,
		// 	input: dto.LinkCreateReq{
		// 		Short:     "",
		// 		Comment:   "test link 1",
		// 		Origin:    "https://www.google.com",
		// 		StartTime: null.TimeFrom(time.Now()),
		// 		EndTime:   null.TimeFrom(time.Now().Add(24 * time.Hour)),
		// 	},
		// 	errorCode: 0,
		// },
		// 测试用例2：正常创建短链接，指定short字段，使用自定义的short值
		{
			userID: 2,
			input: dto.LinkCreateReq{
				Short:     "bing",
				Comment:   "test link 2",
				Origin:    "https://www.bing.com",
				StartTime: null.NewTime(time.Date(2023, 7, 30, 12, 0, 12, 0, time.Local), true),
				EndTime:   null.NewTime(time.Date(2023, 8, 1, 12, 0, 12, 0, time.Local), true),
			},
			errorCode: dto.ErrShortLinkExist,
		},
		// // 测试用例3：创建短链接失败，因为short字段已经存在
		// {
		// 	userID: 3,
		// 	input: dto.LinkCreateReq{
		// 		Short:     "bing",
		// 		Comment:   "test link 3",
		// 		Origin:    "https://www.yahoo.com",
		// 		StartTime: null.TimeFrom(time.Now()),
		// 		EndTime:   null.TimeFrom(time.Now().Add(24 * time.Hour)),
		// 	},
		// 	errorCode: dto.ErrShortLinkExist,
		// },
	}

	testLinkCtl := new(LinkController)
	for _, test := range tests {
		w := httptest.NewRecorder()
		ctx, _ := gin.CreateTestContext(w)
		ctx.Set(model.USER_ID_KEY, test.userID)
		_, err := testLinkCtl.Create(ctx, &test.input)

		if test.errorCode != 0 {
			assert.Equal(t, test.errorCode, stacktrace.GetCode(err))
			continue
		}
		assert.NoError(t, err)
	}

}

func TestDelete(t *testing.T) {

	var tests = []struct {
		userID    uint
		input     dto.LinkDeleteReq
		errorCode stacktrace.ErrorCode
	}{
		// 测试用例1：正常创建短链接，指定short字段，使用自定义的short值
		{
			userID: 2,
			input: dto.LinkDeleteReq{
				Short: "bing",
			},
			errorCode: dto.ErrShortLinkExist,
		},
	}

	testLinkCtl := new(LinkController)
	for _, test := range tests {
		w := httptest.NewRecorder()
		ctx, _ := gin.CreateTestContext(w)
		ctx.Set(model.USER_ID_KEY, test.userID)
		err := testLinkCtl.Delete(ctx, &test.input)

		if test.errorCode != 0 {
			assert.Equal(t, test.errorCode, stacktrace.GetCode(err))
			continue
		}
		assert.NoError(t, err)
	}

}

func TestGetinfo(t *testing.T) {
	var tests = []struct {
		userID    uint
		input     dto.GetLinkInfoReq
		output    dto.GetLinkInfoResp
		errorCode stacktrace.ErrorCode
	}{
		{
			userID: 2,
			input: dto.GetLinkInfoReq{
				Short: "bing",
			},
			output: dto.GetLinkInfoResp{
				Active:    null.BoolFrom(true),
				Comment:   "test link 2",
				EndTime:   null.NewTime(time.Date(2023, 8, 1, 12, 0, 12, 0, time.Local), true),
				Origin:    "https://www.bing.com",
				Short:     "short1", //注意此处和bing不一样
				StartTime: null.NewTime(time.Date(2023, 7, 30, 12, 0, 12, 0, time.Local), true),
			},
			errorCode: 0,
		},
	}
	controller := &LinkController{}
	for _, test := range tests {
		w := httptest.NewRecorder()
		ctx, _ := gin.CreateTestContext(w)
		ctx.Set(model.USER_ID_KEY, test.userID)
		resp, err := controller.GetInfo(ctx, &test.input)

		assert.Equal(t, test.output.Active, resp.Active)
		assert.Equal(t, test.output.Comment, resp.Comment)
		assert.Equal(t, test.output.EndTime, resp.EndTime)
		assert.Equal(t, test.output.Origin, resp.Origin)
		assert.Equal(t, test.output.Short, resp.Short)
		assert.Equal(t, test.output.StartTime, resp.StartTime)
		if test.errorCode != 0 {
			assert.Equal(t, test.errorCode, stacktrace.GetCode(err))

			continue
		}
		assert.NoError(t, err)
	}
}

func TestUpdate(t *testing.T) {
	var tests = []struct {
		userID    uint
		input     dto.LinkUpdateReq
		errorCode stacktrace.ErrorCode
	}{
		// 测试用例1：正常更新短链接，指定short字段搜索短连接并更新为req中的值
		{
			userID: 2,
			input: dto.LinkUpdateReq{
				Short:     "bing",
				Comment:   "test updata 2",
				Origin:    "https://www.bing.com",
				StartTime: null.NewTime(time.Date(2023, 7, 30, 12, 0, 12, 0, time.Local), true),
				EndTime:   null.NewTime(time.Date(2023, 8, 1, 12, 0, 12, 0, time.Local), true),
				Active:    null.BoolFrom(true),
			},
			errorCode: dto.NoErr,
		},
	}

	testLinkCtl := new(LinkController)
	for _, test := range tests {
		w := httptest.NewRecorder()
		ctx, _ := gin.CreateTestContext(w)
		ctx.Set(model.USER_ID_KEY, test.userID)
		err := testLinkCtl.Update(ctx, &test.input)

		if test.errorCode != 0 {
			assert.Equal(t, test.errorCode, stacktrace.GetCode(err))
			continue
		}
		assert.NoError(t, err)
	}

}

func TestRegister(t *testing.T) {
	var tests = []struct {
		userID    uint
		input     dto.UserRegisterReq
		errorCode stacktrace.ErrorCode
	}{
		// 测试用例1：正常创建短链接，指定short字段，使用自定义的short值
		{
			userID: 3,
			input: dto.UserRegisterReq{
				Email:    "3220000000@zju.edu.cn",
				Name:     "3333",
				Password: "123",
			},
			errorCode: dto.NoErr,
		},
	}

	testUserCtl := new(UserController)
	for _, test := range tests {
		w := httptest.NewRecorder()
		ctx, _ := gin.CreateTestContext(w)
		ctx.Set(model.USER_ID_KEY, test.userID)
		err := testUserCtl.Register(ctx, &test.input)

		if test.errorCode != 0 {
			assert.Equal(t, test.errorCode, stacktrace.GetCode(err))
			continue
		}
		assert.NoError(t, err)
	}

}

func TestLogin(t *testing.T) {
	var tests = []struct {
		userID    uint
		input     dto.UserLoginReq
		errorCode stacktrace.ErrorCode
	}{
		// 测试用例1：密码错误
		{
			userID: 2,
			input: dto.UserLoginReq{
				CAPTCHAID:    "1",
				CAPTCHAValue: "1",
				Email:        "3220106025@zju.edu.cn",
				Password:     "1234",
			},
			errorCode: dto.NoErr,
		},
	}

	testUserCtl := new(UserController)
	for _, test := range tests {
		w := httptest.NewRecorder()
		ctx, _ := gin.CreateTestContext(w)
		ctx.Set(model.USER_ID_KEY, test.userID)
		err := testUserCtl.Login(ctx, &test.input)

		if test.errorCode != 0 {
			assert.Equal(t, test.errorCode, stacktrace.GetCode(err))
			continue
		}
		assert.NoError(t, err)
	}

}

// 测试修改信息函数
func TestUserModifyInfoReq(t *testing.T) {
	var tests = []struct {
		userID    uint
		input     dto.UserModifyInfoReq
		errorCode stacktrace.ErrorCode
	}{
		// 测试用例1：修改邮箱
		{
			userID: 1,
			input: dto.UserModifyInfoReq{
				Email: "3220406011@thu.edu.cn",
				ID:    1,
				Name:  "thu",
			},
			errorCode: dto.NoErr,
		},
	}

	testUserCtl := new(UserController)
	for _, test := range tests {
		w := httptest.NewRecorder()
		ctx, _ := gin.CreateTestContext(w)
		ctx.Set(model.USER_ID_KEY, test.userID)
		err := testUserCtl.ModifyInfo(ctx, &test.input)

		if test.errorCode != 0 {
			assert.Equal(t, test.errorCode, stacktrace.GetCode(err))
			continue
		}
		assert.NoError(t, err)
	}

}

func TestUserModifyPwdReq(t *testing.T) {
	var tests = []struct {
		userID    uint
		input     dto.UserModifyPwdReq
		errorCode stacktrace.ErrorCode
	}{
		// 测试用例1：旧密码不正确
		{
			userID: 3,
			input: dto.UserModifyPwdReq{
				NewPwd: "111111",
				OldPwd: "123",
			},
			errorCode: dto.NoErr,
		},
	}

	testUserCtl := new(UserController)
	for _, test := range tests {
		w := httptest.NewRecorder()
		ctx, _ := gin.CreateTestContext(w)
		ctx.Set(model.USER_ID_KEY, test.userID)
		err := testUserCtl.ModifyPwd(ctx, &test.input)

		if test.errorCode != 0 {
			assert.Equal(t, test.errorCode, stacktrace.GetCode(err))
			continue
		}
		assert.NoError(t, err)
	}

}
