package controller

import (
	"go-svc-tpl/api/dto"
	"go-svc-tpl/internal/dao"
	"go-svc-tpl/internal/dao/model"
	"go-svc-tpl/utils/stacktrace"

	"crypto/md5"
	"encoding/hex"

	"github.com/gin-gonic/gin"
	"github.com/sirupsen/logrus"
	"gopkg.in/guregu/null.v4"
	"gorm.io/gorm"
)

const LinkTable = "Links"

// >>>>>>>>>>>>>>>>>> Interface  >>>>>>>>>>>>>>>>>>
type ILinkController interface { //定义tag link 下的操作
	Create(*gin.Context, *dto.LinkCreateReq) (*dto.LinkCreateResp, error)
	Delete(*gin.Context, *dto.LinkDeleteReq) error
	GetInfo(*gin.Context, *dto.GetLinkInfoReq) (*dto.GetLinkInfoResp, error)
	Update(*gin.Context, *dto.LinkUpdateReq) error
	GetList(*gin.Context, *dto.GetLinkListReq) (*dto.GetLinkListResp, error)
}

// >>>>>>>>>>>>>>>>>> Controller >>>>>>>>>>>>>>>>>>

// check interface implementation
var _ ILinkController = (*LinkController)(nil)

var NewLinkController = func() *LinkController {
	return &LinkController{}
}

type LinkController struct {
}

// create
// 用gorm创建短链接
// short 如果为空字符串，则自动生成链接
// 如果起止时间为 null 则表示不设置起止时间
func (c *LinkController) Create(ctx *gin.Context, req *dto.LinkCreateReq) (*dto.LinkCreateResp, error) {
	userID := ctx.GetUint(model.USER_ID_KEY)
	newLink := &model.Link{
		Short:     req.Short,
		Comment:   req.Comment,
		Origin:    req.Origin,
		StartTime: req.StartTime,
		EndTime:   req.EndTime,
		Active:    null.BoolFrom(true),
		OwnerID:   userID,
	}
	if newLink.Short == "" {
		newLink.Short = GenerateShort(newLink.Origin)
	}

	var link model.Link
	err := dao.DB(ctx).Table("Links").Where(&model.Link{Short: newLink.Short}).First(&link).Error
	if err != nil {
		// 没找到相关记录 说明不是短连接重复
		if err == gorm.ErrRecordNotFound {
			//logrus.Fatal(err)
			//存到数据库中
			err = dao.DB(ctx).Create(newLink).Error
			if err != nil {
				logrus.Error("Internal Error.")
				return nil, stacktrace.PropagateWithCode(err, dto.InternalError, "InternalError")
			}
			// 没有错误 正常创建
			return &dto.LinkCreateResp{
				Short:     newLink.Short,
				Origin:    newLink.Origin,
				Comment:   newLink.Comment,
				StartTime: newLink.StartTime,
				EndTime:   newLink.EndTime,
				Active:    newLink.Active,
			}, nil
		}
		logrus.Error("Internal Error.")
		return nil, stacktrace.PropagateWithCode(err, dto.InternalError, "InternalError")
	}
	//如果找到了的话
	return nil, stacktrace.PropagateWithCode(err, dto.ErrShortLinkExist, "ErrShortLinkExist")
}

// delete
func (c *LinkController) Delete(ctx *gin.Context, req *dto.LinkDeleteReq) error {
	//userID := ctx.GetUint(model.USER_ID_KEY)
	deleteLink := &model.Link{
		Short: req.Short,
	}
	err := dao.DB(ctx).Table("Links").Where(&model.Link{Short: deleteLink.Short}).Delete(deleteLink).Error
	if err != nil {
		logrus.Fatal(err)
		return err
	}
	var link model.Link
	err = dao.DB(ctx).Where(&model.Link{Short: deleteLink.Short}).First(&link).Error
	if err != nil {
		if err == gorm.ErrRecordNotFound {
			logrus.Fatal(err)
			return stacktrace.PropagateWithCode(err, dto.ErrNoShortLink, "ErrNoShortLink")
		}
		return err
	}
	return nil
}

// getinfo
func (c *LinkController) GetInfo(ctx *gin.Context, req *dto.GetLinkInfoReq) (*dto.GetLinkInfoResp, error) {
	userID := ctx.GetUint(model.USER_ID_KEY)
	getinfoLink := &model.Link{
		Short:   req.Short,
		OwnerID: userID,
	}

	var link model.Link
	err := dao.DB(ctx).Table("Links").Where(&model.Link{Short: getinfoLink.Short}).First(&link).Error
	if err != nil {
		if err == gorm.ErrRecordNotFound {
			logrus.Fatal(err)
			return nil, stacktrace.PropagateWithCode(err, dto.ErrNoShortLink, "ErrNoShortLink")
		}
		return nil, err
	}
	return &dto.GetLinkInfoResp{
		Short:     link.Short,
		Origin:    link.Origin,
		Comment:   link.Comment,
		StartTime: link.StartTime,
		EndTime:   link.EndTime,
		Active:    link.Active,
	}, nil
}

// update
func (c *LinkController) Update(ctx *gin.Context, req *dto.LinkUpdateReq) error {
	userID := ctx.GetUint(model.USER_ID_KEY)
	updateLink := &model.Link{
		Short:     req.Short,
		Origin:    req.Origin,
		Comment:   req.Comment,
		StartTime: req.StartTime,
		EndTime:   req.EndTime,
		Active:    req.Active,
		OwnerID:   userID,
	}
	err := dao.DB(ctx).Table("Links").Where(&model.Link{Short: updateLink.Short}).Updates(&updateLink).Error
	if err != nil {
		logrus.Fatal(err)
		return stacktrace.PropagateWithCode(err, dto.ErrShortLinkExist, "ErrShortLinkExist")
	}
	return nil
}

// getlist
func (c *LinkController) GetList(ctx *gin.Context, req *dto.GetLinkListReq) (*dto.GetLinkListResp, error) {
	var links []dto.ShortLinkModel
	var total int64

	query := dao.DB(ctx).Model(&dto.ShortLinkModel{})

	if req.PageNumber > 0 && req.PageSize > 0 {
		offset := (req.PageNumber - 1) * req.PageSize
		query = query.Offset(int(offset)).Limit(int(req.PageSize))
	}
	if result := query.Find(&links); result.Error != nil {
		return nil, result.Error
	}
	query.Count(&total)

	resp := &dto.GetLinkListResp{
		Links: links,
		Total: total,
	}

	return resp, nil
}

// 生成短链接
// 输入长链接（字符串） 返回短链接（字符串）
func GenerateShort(origin string) string {
	// 使用MD5哈希函数对长链接进行哈希计算
	hash := md5.Sum([]byte(origin))
	// 将哈希结果转换为16进制字符串
	hashString := hex.EncodeToString(hash[:])
	// 取哈希结果的前8个字符作为短链接
	short := hashString[:8]
	return short
}
