package main

import (
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"

	"github.com/spf13/viper"
	"gorm.io/driver/postgres"
	"gorm.io/gorm"
)

type Response struct {
	Code int    `json:"code"`
	Msg  string `json:"msg"`
	Data any    `json:"data"`
}

type Comment struct {
	ID      int    `json:"id" gorm:"primaryKey"`
	Name    string `json:"name"`
	Content string `json:"content"`
}

var db *gorm.DB

func init() {

	viper.SetConfigName("config")
	viper.SetConfigType("yaml")
	viper.AddConfigPath(".")
	viper.ReadInConfig()

	host := viper.GetString("db.host")
	user := viper.GetString("db.user")
	password := viper.GetString("db.password")
	dbname := viper.GetString("db.dbname")
	port := viper.GetString("db.port")
	sslmode := viper.GetString("db.sslmode")
	timezone := viper.GetString("db.timezone")

	dsn := fmt.Sprintf("host=%s user=%s password=%s dbname=%s port=%s sslmode=%s TimeZone=%s",
		host, user, password, dbname, port, sslmode, timezone)

	var err error
	db, err = gorm.Open(postgres.Open(dsn), &gorm.Config{})
	if err != nil {
		panic("failed to connect database")
	}

	// 自动迁移 Comment 表
	if err := db.AutoMigrate(&Comment{}); err != nil {
		panic("failed to migrate database")
	}
}

func enableCors(w http.ResponseWriter) {
	w.Header().Set("Access-Control-Allow-Origin", "*")
	w.Header().Set("Access-Control-Allow-Methods", "GET, POST, DELETE, OPTIONS")
	w.Header().Set("Access-Control-Allow-Headers", "Content-Type, Authorization")
}

func getHandler(w http.ResponseWriter, r *http.Request) {
	enableCors(w)

	page := r.URL.Query().Get("page")
	size := r.URL.Query().Get("size")

	// Convert page and size to integers
	pageInt, err := strconv.Atoi(page)
	if err != nil || pageInt < 1 {
		pageInt = 1
	}
	sizeInt, err := strconv.Atoi(size)
	if err != nil || sizeInt < 1 {
		sizeInt = 10
	}

	// Fetch comments from the database with pagination
	var comments []Comment
	var total int64
	db.Model(&Comment{}).Count(&total)
	db.Order("id desc").Offset((pageInt - 1) * sizeInt).Limit(sizeInt).Find(&comments)

	response := Response{
		Code: 0,
		Msg:  "success",
		Data: map[string]interface{}{
			"total":    total,
			"comments": comments,
		},
	}

	w.Header().Set("Content-Type", "application/json")
	if err := json.NewEncoder(w).Encode(response); err != nil {
		http.Error(w, "Failed to encode response", http.StatusInternalServerError)
		return
	}
}

func addHandler(w http.ResponseWriter, r *http.Request) {
	enableCors(w)

	w.Header().Set("Content-Type", "application/json")
	var comment Comment
	json.NewDecoder(r.Body).Decode(&comment)

	if comment.Name == "" || comment.Content == "" {
		return
	}

	db.Create(&comment)

	response := Response{
		Code: 0,
		Msg:  "success",
		Data: comment,
	}

	json.NewEncoder(w).Encode(response)
}

func deleteHandler(w http.ResponseWriter, r *http.Request) {
	enableCors(w)

	w.Header().Set("Content-Type", "application/json")
	id := r.URL.Query().Get("id")
	idInt, err := strconv.Atoi(id)
	if err != nil {
		http.Error(w, "Invalid ID", http.StatusBadRequest)
		return
	}
	db.Delete(&Comment{}, idInt)

	response := Response{
		Code: 0,
		Msg:  "success",
		Data: nil,
	}

	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(response)
}

func main() {
	http.HandleFunc("/comment/get", getHandler)
	http.HandleFunc("/comment/add", addHandler)
	http.HandleFunc("/comment/delete", deleteHandler)
	http.ListenAndServe(":8080", nil)
}
