import bcrypt from 'bcryptjs'
import jwt from 'jsonwebtoken'
import { prisma } from './prisma'
import { User } from '@/types/api'

const JWT_SECRET = process.env.JWT_SECRET || 'your-fallback-secret'

export interface JWTPayload {
  userId: number;
  username: string;
}

// 密码哈希
export async function hashPassword(password: string): Promise<string> {
  return await bcrypt.hash(password, 12)
}

// 验证密码
export async function verifyPassword(password: string, hashedPassword: string): Promise<boolean> {
  return await bcrypt.compare(password, hashedPassword)
}

// 生成 JWT Token
export function generateToken(user: { id: number; username: string }): string {
  const payload: JWTPayload = {
    userId: user.id,
    username: user.username
  }
  return jwt.sign(payload, JWT_SECRET, { expiresIn: '7d' })
}

// 验证 JWT Token
export function verifyToken(token: string): JWTPayload | null {
  try {
    return jwt.verify(token, JWT_SECRET) as JWTPayload
  } catch (error) {
    return null
  }
}

// 从请求中获取用户信息
export async function getUserFromRequest(request: Request): Promise<User | null> {
  const authHeader = request.headers.get('authorization')
  if (!authHeader || !authHeader.startsWith('Bearer ')) {
    return null
  }

  const token = authHeader.substring(7)
  const payload = verifyToken(token)
  
  if (!payload) {
    return null
  }

  try {
    const user = await prisma.user.findUnique({
      where: { id: payload.userId },
      select: { id: true, username: true }
    })
    return user
  } catch (error) {
    return null
  }
}

// 注册用户
export async function registerUser(username: string, password: string) {
  // 检查用户名是否已存在
  const existingUser = await prisma.user.findUnique({
    where: { username }
  })

  if (existingUser) {
    throw new Error('用户名已存在')
  }

  // 创建新用户
  const hashedPassword = await hashPassword(password)
  const user = await prisma.user.create({
    data: {
      username,
      password: hashedPassword
    },
    select: {
      id: true,
      username: true
    }
  })

  return user
}

// 登录用户
export async function loginUser(username: string, password: string) {
  const user = await prisma.user.findUnique({
    where: { username }
  })

  if (!user || !(await verifyPassword(password, user.password))) {
    throw new Error('用户名或密码错误')
  }

  return {
    id: user.id,
    username: user.username
  }
}
