import { 
  ApiResponse, 
  LoginRequest, 
  RegisterRequest, 
  RoomAddArgs, 
  RoomDeleteArgs, 
  MessageAddArgs,
  RoomAddRes,
  RoomListRes,
  RoomMessageListRes,
  RoomMessageGetUpdateRes
} from '@/types/api'

const API_BASE = ''

// 获取存储的 token
export function getToken(): string | null {
  if (typeof window === 'undefined') return null
  return localStorage.getItem('token')
}

// 设置 token
export function setToken(token: string): void {
  if (typeof window !== 'undefined') {
    localStorage.setItem('token', token)
  }
}

// 清除 token
export function clearToken(): void {
  if (typeof window !== 'undefined') {
    localStorage.removeItem('token')
  }
}

// 通用请求函数
async function apiRequest<T>(
  endpoint: string, 
  options: RequestInit = {}
): Promise<ApiResponse<T>> {
  const token = getToken()
  
  const headers: HeadersInit = {
    'Content-Type': 'application/json',
    ...(options.headers || {})
  }
  
  if (token) {
    headers['Authorization'] = `Bearer ${token}`
  }

  const response = await fetch(`${API_BASE}${endpoint}`, {
    ...options,
    headers
  })

  const data: ApiResponse<T> = await response.json()
  
  if (data.code !== 0) {
    throw new Error(data.message || '请求失败')
  }
  
  return data
}

// 认证相关 API
export const authApi = {
  async login(credentials: LoginRequest) {
    return await apiRequest<{ user: any; token: string }>('/api/auth/login', {
      method: 'POST',
      body: JSON.stringify(credentials)
    })
  },

  async register(credentials: RegisterRequest) {
    return await apiRequest<{ user: any; token: string }>('/api/auth/register', {
      method: 'POST',
      body: JSON.stringify(credentials)
    })
  }
}

// 房间相关 API
export const roomApi = {
  async addRoom(args: RoomAddArgs) {
    return await apiRequest<RoomAddRes>('/api/room/add', {
      method: 'POST',
      body: JSON.stringify(args)
    })
  },

  async getRoomList() {
    return await apiRequest<RoomListRes>('/api/room/list')
  },

  async deleteRoom(args: RoomDeleteArgs) {
    return await apiRequest<null>('/api/room/delete', {
      method: 'POST',
      body: JSON.stringify(args)
    })
  }
}

// 消息相关 API
export const messageApi = {
  async addMessage(args: MessageAddArgs) {
    return await apiRequest<null>('/api/message/add', {
      method: 'POST',
      body: JSON.stringify(args)
    })
  },

  async getRoomMessages(roomId: number) {
    return await apiRequest<RoomMessageListRes>(`/api/room/message/list?roomId=${roomId}`)
  },

  async getMessageUpdates(roomId: number, sinceMessageId: number) {
    return await apiRequest<RoomMessageGetUpdateRes>(
      `/api/room/message/getUpdate?roomId=${roomId}&sinceMessageId=${sinceMessageId}`
    )
  }
}
