'use client'

import { useState, useEffect, useRef } from 'react'
import useSWR, { mutate } from 'swr'
import { useAuth } from '@/contexts/AuthContext'
import { roomApi, messageApi } from '@/lib/api'
import { RoomPreviewInfo, Message } from '@/types/api'

const ChatRoom = () => {
  const { user, nickname, logout } = useAuth()
  const [selectedRoomId, setSelectedRoomId] = useState<number | null>(null)
  const [messageInput, setMessageInput] = useState('')
  const [showAddRoomModal, setShowAddRoomModal] = useState(false)
  const [newRoomName, setNewRoomName] = useState('')
  const [contextMenu, setContextMenu] = useState<{
    x: number
    y: number
    roomId: number
  } | null>(null)
  const messagesEndRef = useRef<HTMLDivElement>(null)

  // 获取房间列表
  const { data: roomsData, error: roomsError } = useSWR(
    'rooms',
    () => roomApi.getRoomList().then(res => res.data)
  )

  // 获取选中房间的消息
  const { data: messagesData } = useSWR(
    selectedRoomId ? `messages-${selectedRoomId}` : null,
    () => selectedRoomId ? messageApi.getRoomMessages(selectedRoomId).then(res => res.data) : null
  )

  // 自动滚动到最新消息
  useEffect(() => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' })
  }, [messagesData?.messages])

  // 实时更新消息 - 每秒检查一次
  useEffect(() => {
    if (!selectedRoomId || !messagesData?.messages.length) return

    const interval = setInterval(async () => {
      try {
        const lastMessageId = messagesData.messages[messagesData.messages.length - 1].messageId
        const updateRes = await messageApi.getMessageUpdates(selectedRoomId, lastMessageId)
        
        if (updateRes.data.messages.length > 0) {
          // 有新消息，刷新消息列表
          mutate(`messages-${selectedRoomId}`)
          // 同时刷新房间列表以更新最后一条消息
          mutate('rooms')
        }
      } catch (error) {
        console.error('检查消息更新失败:', error)
      }
    }, 1000)

    return () => clearInterval(interval)
  }, [selectedRoomId, messagesData?.messages])

  // 关闭右键菜单
  useEffect(() => {
    const handleClick = () => setContextMenu(null)
    document.addEventListener('click', handleClick)
    return () => document.removeEventListener('click', handleClick)
  }, [])

  const handleSendMessage = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!messageInput.trim() || !selectedRoomId || !nickname) return

    try {
      await messageApi.addMessage({
        roomId: selectedRoomId,
        content: messageInput.trim(),
        sender: nickname
      })
      setMessageInput('')
      // 刷新消息列表和房间列表
      mutate(`messages-${selectedRoomId}`)
      mutate('rooms')
    } catch (error) {
      alert('发送消息失败: ' + (error instanceof Error ? error.message : '未知错误'))
    }
  }

  const handleAddRoom = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!newRoomName.trim() || !nickname) return

    try {
      await roomApi.addRoom({
        user: nickname,
        roomName: newRoomName.trim()
      })
      setNewRoomName('')
      setShowAddRoomModal(false)
      mutate('rooms')
    } catch (error) {
      alert('创建房间失败: ' + (error instanceof Error ? error.message : '未知错误'))
    }
  }

  const handleDeleteRoom = async (roomId: number) => {
    if (!nickname) return
    
    if (confirm('确定要删除这个房间吗？')) {
      try {
        await roomApi.deleteRoom({
          user: nickname,
          roomId
        })
        if (selectedRoomId === roomId) {
          setSelectedRoomId(null)
        }
        mutate('rooms')
        setContextMenu(null)
      } catch (error) {
        alert('删除房间失败: ' + (error instanceof Error ? error.message : '未知错误'))
      }
    }
  }

  const handleRoomRightClick = (e: React.MouseEvent, roomId: number) => {
    e.preventDefault()
    setContextMenu({
      x: e.clientX,
      y: e.clientY,
      roomId
    })
  }

  const formatTime = (timestamp: number) => {
    return new Date(timestamp).toLocaleTimeString('zh-CN', {
      hour: '2-digit',
      minute: '2-digit'
    })
  }

  const getAvatarText = (username: string) => {
    return username.charAt(0).toUpperCase()
  }

  const selectedRoom = roomsData?.rooms.find(room => room.roomId === selectedRoomId)

  return (
    <div className="chat-container">
      {/* 侧边栏 - 房间列表 */}
      <div className="sidebar">
        <div className="sidebar-header">
          <h2>聊天室</h2>
          <button className="logout-btn" onClick={logout}>
            退出
          </button>
        </div>
        
        <div className="room-list">
          {roomsData?.rooms.map((room: RoomPreviewInfo) => (
            <div
              key={room.roomId}
              className={`room-item ${selectedRoomId === room.roomId ? 'active' : ''}`}
              onClick={() => setSelectedRoomId(room.roomId)}
              onContextMenu={(e) => handleRoomRightClick(e, room.roomId)}
            >
              <div className="room-name">{room.roomName}</div>
              {room.lastMessage && (
                <div className="room-last-message">
                  {room.lastMessage.sender}: {room.lastMessage.content}
                </div>
              )}
            </div>
          ))}
        </div>

        <button 
          className="add-room-btn"
          onClick={() => setShowAddRoomModal(true)}
        >
          + 添加房间
        </button>
      </div>

      {/* 聊天区域 */}
      <div className="chat-area">
        {selectedRoom ? (
          <>
            <div className="chat-header">
              <h3>{selectedRoom.roomName}</h3>
            </div>

            <div className="messages-container">
              {messagesData?.messages.map((message: Message) => (
                <div key={message.messageId} className="message">
                  <div className="message-avatar">
                    {getAvatarText(message.sender)}
                  </div>
                  <div className="message-content">
                    <div className="message-header">
                      <span className="message-sender">{message.sender}</span>
                      <span className="message-time">{formatTime(message.time)}</span>
                    </div>
                    <div className="message-text">{message.content}</div>
                  </div>
                </div>
              ))}
              <div ref={messagesEndRef} />
            </div>

            <div className="message-input-container">
              <form className="message-input-form" onSubmit={handleSendMessage}>
                <input
                  type="text"
                  className="message-input"
                  placeholder="输入消息..."
                  value={messageInput}
                  onChange={(e) => setMessageInput(e.target.value)}
                />
                <button 
                  type="submit" 
                  className="send-btn"
                  disabled={!messageInput.trim()}
                >
                  发送
                </button>
              </form>
            </div>
          </>
        ) : (
          <div className="no-room-selected">
            选择一个房间开始聊天
          </div>
        )}
      </div>

      {/* 添加房间模态框 */}
      {showAddRoomModal && (
        <div className="modal-overlay">
          <div className="modal">
            <h3>创建新房间</h3>
            <form onSubmit={handleAddRoom}>
              <div className="form-group">
                <label htmlFor="roomName">房间名称</label>
                <input
                  id="roomName"
                  type="text"
                  value={newRoomName}
                  onChange={(e) => setNewRoomName(e.target.value)}
                  placeholder="请输入房间名称"
                  maxLength={50}
                />
              </div>
              <div className="modal-buttons">
                <button 
                  type="button" 
                  className="btn btn-secondary"
                  onClick={() => {
                    setShowAddRoomModal(false)
                    setNewRoomName('')
                  }}
                >
                  取消
                </button>
                <button type="submit" className="btn">
                  创建
                </button>
              </div>
            </form>
          </div>
        </div>
      )}

      {/* 右键菜单 */}
      {contextMenu && (
        <div 
          className="context-menu"
          style={{
            position: 'fixed',
            left: contextMenu.x,
            top: contextMenu.y
          }}
        >
          <div 
            className="context-menu-item danger"
            onClick={() => handleDeleteRoom(contextMenu.roomId)}
          >
            删除房间
          </div>
        </div>
      )}
    </div>
  )
}

export default ChatRoom
