import { NextRequest, NextResponse } from 'next/server'
import { prisma } from '@/lib/prisma'
import { ApiResponse, RoomMessageListRes, Message } from '@/types/api'

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const roomIdStr = searchParams.get('roomId')
    
    if (!roomIdStr) {
      return NextResponse.json({
        message: '房间 ID 不能为空',
        code: 1,
        data: null
      } as ApiResponse, { status: 400 })
    }

    const roomId = parseInt(roomIdStr)
    if (isNaN(roomId)) {
      return NextResponse.json({
        message: '房间 ID 格式错误',
        code: 1,
        data: null
      } as ApiResponse, { status: 400 })
    }

    // 检查房间是否存在
    const room = await prisma.room.findUnique({
      where: { id: roomId }
    })

    if (!room) {
      return NextResponse.json({
        message: '房间不存在',
        code: 1,
        data: null
      } as ApiResponse, { status: 404 })
    }

    // 获取房间消息
    const messages = await prisma.message.findMany({
      where: { roomId },
      include: {
        sender: {
          select: {
            username: true
          }
        }
      },
      orderBy: {
        createdAt: 'asc'
      }
    })

    const formattedMessages: Message[] = messages.map(msg => ({
      messageId: msg.id,
      roomId: msg.roomId,
      sender: msg.sender.username,
      content: msg.content,
      time: msg.createdAt.getTime()
    }))

    return NextResponse.json({
      message: '获取消息列表成功',
      code: 0,
      data: {
        messages: formattedMessages
      } as RoomMessageListRes
    } as ApiResponse)
  } catch (error) {
    console.error('获取消息列表失败:', error)
    return NextResponse.json({
      message: '获取消息列表失败',
      code: 1,
      data: null
    } as ApiResponse, { status: 500 })
  }
}
