import { NextRequest, NextResponse } from 'next/server'
import { prisma } from '@/lib/prisma'
import { ApiResponse, RoomListRes, RoomPreviewInfo } from '@/types/api'

export async function GET(request: NextRequest) {
  try {
    // 获取所有房间及其最新消息
    const rooms = await prisma.room.findMany({
      include: {
        messages: {
          orderBy: {
            createdAt: 'desc'
          },
          take: 1,
          include: {
            sender: {
              select: {
                username: true
              }
            }
          }
        },
        creator: {
          select: {
            username: true
          }
        }
      },
      orderBy: {
        createdAt: 'desc'
      }
    })

    const roomPreviews: RoomPreviewInfo[] = rooms.map(room => ({
      roomId: room.id,
      roomName: room.name,
      lastMessage: room.messages.length > 0 ? {
        messageId: room.messages[0].id,
        roomId: room.messages[0].roomId,
        sender: room.messages[0].sender.username,
        content: room.messages[0].content,
        time: room.messages[0].createdAt.getTime()
      } : null
    }))

    return NextResponse.json({
      message: '获取房间列表成功',
      code: 0,
      data: {
        rooms: roomPreviews
      } as RoomListRes
    } as ApiResponse)
  } catch (error) {
    console.error('获取房间列表失败:', error)
    return NextResponse.json({
      message: '获取房间列表失败',
      code: 1,
      data: null
    } as ApiResponse, { status: 500 })
  }
}
