import { NextRequest, NextResponse } from 'next/server'
import { getUserFromRequest } from '@/lib/auth'
import { prisma } from '@/lib/prisma'
import { RoomDeleteArgs, ApiResponse } from '@/types/api'

export async function POST(request: NextRequest) {
  try {
    // 验证用户身份
    const user = await getUserFromRequest(request)
    if (!user) {
      return NextResponse.json({
        message: '未登录或登录已过期',
        code: 1,
        data: null
      } as ApiResponse, { status: 401 })
    }

    const body: RoomDeleteArgs = await request.json()
    const { roomId } = body

    if (!roomId) {
      return NextResponse.json({
        message: '房间 ID 不能为空',
        code: 1,
        data: null
      } as ApiResponse, { status: 400 })
    }

    // 检查房间是否存在以及用户是否为创建者
    const room = await prisma.room.findUnique({
      where: { id: roomId }
    })

    if (!room) {
      return NextResponse.json({
        message: '房间不存在',
        code: 1,
        data: null
      } as ApiResponse, { status: 404 })
    }

    if (room.creatorId !== user.id) {
      return NextResponse.json({
        message: '只有房间创建者可以删除房间',
        code: 1,
        data: null
      } as ApiResponse, { status: 403 })
    }

    // 删除房间（会级联删除相关消息）
    await prisma.room.delete({
      where: { id: roomId }
    })

    return NextResponse.json({
      message: '房间删除成功',
      code: 0,
      data: null
    } as ApiResponse)
  } catch (error) {
    console.error('删除房间失败:', error)
    return NextResponse.json({
      message: '删除房间失败',
      code: 1,
      data: null
    } as ApiResponse, { status: 500 })
  }
}
