import { NextRequest, NextResponse } from 'next/server'
import { getUserFromRequest } from '@/lib/auth'
import { prisma } from '@/lib/prisma'
import { MessageAddArgs, ApiResponse } from '@/types/api'

export async function POST(request: NextRequest) {
  try {
    // 验证用户身份
    const user = await getUserFromRequest(request)
    if (!user) {
      return NextResponse.json({
        message: '未登录或登录已过期',
        code: 1,
        data: null
      } as ApiResponse, { status: 401 })
    }

    const body: MessageAddArgs = await request.json()
    const { roomId, content } = body

    if (!roomId || !content || content.trim().length === 0) {
      return NextResponse.json({
        message: '房间 ID 和消息内容不能为空',
        code: 1,
        data: null
      } as ApiResponse, { status: 400 })
    }

    // 检查房间是否存在
    const room = await prisma.room.findUnique({
      where: { id: roomId }
    })

    if (!room) {
      return NextResponse.json({
        message: '房间不存在',
        code: 1,
        data: null
      } as ApiResponse, { status: 404 })
    }

    // 创建消息
    await prisma.message.create({
      data: {
        content: content.trim(),
        roomId,
        senderId: user.id
      }
    })

    return NextResponse.json({
      message: '消息发送成功',
      code: 0,
      data: null
    } as ApiResponse)
  } catch (error) {
    console.error('发送消息失败:', error)
    return NextResponse.json({
      message: '发送消息失败',
      code: 1,
      data: null
    } as ApiResponse, { status: 500 })
  }
}
