'use client'

import React, { createContext, useContext, useState, useEffect } from 'react'
import { User } from '@/types/api'
import { getToken, setToken, clearToken, authApi } from '@/lib/api'

interface AuthContextType {
  user: User | null
  nickname: string | null
  loading: boolean
  login: (username: string, password: string) => Promise<void>
  register: (username: string, password: string) => Promise<void>
  logout: () => void
  setNickname: (nickname: string) => void
}

const AuthContext = createContext<AuthContextType | undefined>(undefined)

export function AuthProvider({ children }: { children: React.ReactNode }) {
  const [user, setUser] = useState<User | null>(null)
  const [nickname, setNicknameState] = useState<string | null>(null)
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    // 检查本地存储的认证状态
    const token = getToken()
    const storedNickname = localStorage.getItem('nickname')
    
    if (storedNickname) {
      setNicknameState(storedNickname)
    }

    // TODO: 验证 token 有效性
    setLoading(false)
  }, [])

  const login = async (username: string, password: string) => {
    try {
      const response = await authApi.login({ username, password })
      setUser(response.data.user)
      setToken(response.data.token)
    } catch (error) {
      throw error
    }
  }

  const register = async (username: string, password: string) => {
    try {
      const response = await authApi.register({ username, password })
      setUser(response.data.user)
      setToken(response.data.token)
    } catch (error) {
      throw error
    }
  }

  const logout = () => {
    setUser(null)
    setNicknameState(null)
    clearToken()
    localStorage.removeItem('nickname')
  }

  const setNickname = (newNickname: string) => {
    setNicknameState(newNickname)
    localStorage.setItem('nickname', newNickname)
  }

  return (
    <AuthContext.Provider value={{
      user,
      nickname,
      loading,
      login,
      register,
      logout,
      setNickname
    }}>
      {children}
    </AuthContext.Provider>
  )
}

export function useAuth() {
  const context = useContext(AuthContext)
  if (context === undefined) {
    throw new Error('useAuth must be used within an AuthProvider')
  }
  return context
}
