'use client'

import { useState, useEffect } from 'react'
import { useAuth } from '@/contexts/AuthContext'
import ChatRoom from '@/components/ChatRoom'

export default function Home() {
  const { user, nickname, setNickname } = useAuth()
  const [currentView, setCurrentView] = useState<'auth' | 'setName' | 'chat'>('auth')
  const [isLogin, setIsLogin] = useState(true)
  const [formData, setFormData] = useState({
    username: '',
    password: ''
  })
  const [nicknameInput, setNicknameInput] = useState('')
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState('')

  const { login, register } = useAuth()

  useEffect(() => {
    if (user && nickname) {
      setCurrentView('chat')
    } else if (user && !nickname) {
      setCurrentView('setName')
    } else {
      setCurrentView('auth')
    }
  }, [user, nickname])

  const handleAuth = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!formData.username || !formData.password) {
      setError('用户名和密码不能为空')
      return
    }

    setLoading(true)
    setError('')

    try {
      if (isLogin) {
        await login(formData.username, formData.password)
      } else {
        await register(formData.username, formData.password)
      }
    } catch (err) {
      setError(err instanceof Error ? err.message : '操作失败')
    } finally {
      setLoading(false)
    }
  }

  const handleSetNickname = (e: React.FormEvent) => {
    e.preventDefault()
    if (!nicknameInput.trim()) {
      setError('昵称不能为空')
      return
    }
    setNickname(nicknameInput.trim())
  }

  if (currentView === 'chat') {
    return <ChatRoom />
  }

  if (currentView === 'setName') {
    return (
      <div className="auth-container">
        <form className="auth-form" onSubmit={handleSetNickname}>
          <h1>设置昵称</h1>
          <div className="form-group">
            <label htmlFor="nickname">请输入您的聊天昵称</label>
            <input
              id="nickname"
              type="text"
              value={nicknameInput}
              onChange={(e) => setNicknameInput(e.target.value)}
              placeholder="输入昵称"
              maxLength={20}
            />
          </div>
          {error && <div className="error-message">{error}</div>}
          <button type="submit" className="btn">
            进入聊天室
          </button>
        </form>
      </div>
    )
  }

  return (
    <div className="auth-container">
      <form className="auth-form" onSubmit={handleAuth}>
        <h1>{isLogin ? '登录' : '注册'}</h1>
        <div className="form-group">
          <label htmlFor="username">用户名</label>
          <input
            id="username"
            type="text"
            value={formData.username}
            onChange={(e) => setFormData(prev => ({ ...prev, username: e.target.value }))}
            placeholder="请输入用户名"
          />
        </div>
        <div className="form-group">
          <label htmlFor="password">密码</label>
          <input
            id="password"
            type="password"
            value={formData.password}
            onChange={(e) => setFormData(prev => ({ ...prev, password: e.target.value }))}
            placeholder="请输入密码"
          />
        </div>
        {error && <div className="error-message">{error}</div>}
        <button type="submit" className="btn" disabled={loading}>
          {loading ? '处理中...' : (isLogin ? '登录' : '注册')}
        </button>
        <div className="auth-switch">
          {isLogin ? '没有账号？' : '已有账号？'}
          <button
            type="button"
            onClick={() => {
              setIsLogin(!isLogin)
              setError('')
            }}
          >
            {isLogin ? '立即注册' : '立即登录'}
          </button>
        </div>
      </form>
    </div>
  )
}
