import { NextRequest, NextResponse } from 'next/server'
import { getUserFromRequest } from '@/lib/auth'
import { prisma } from '@/lib/prisma'
import { RoomAddArgs, ApiResponse, RoomAddRes } from '@/types/api'

export async function POST(request: NextRequest) {
  try {
    // 验证用户身份
    const user = await getUserFromRequest(request)
    if (!user) {
      return NextResponse.json({
        message: '未登录或登录已过期',
        code: 1,
        data: null
      } as ApiResponse, { status: 401 })
    }

    const body: RoomAddArgs = await request.json()
    const { roomName } = body

    if (!roomName || roomName.trim().length === 0) {
      return NextResponse.json({
        message: '房间名称不能为空',
        code: 1,
        data: null
      } as ApiResponse, { status: 400 })
    }

    // 创建房间
    const room = await prisma.room.create({
      data: {
        name: roomName.trim(),
        creatorId: user.id
      }
    })

    return NextResponse.json({
      message: '房间创建成功',
      code: 0,
      data: {
        roomId: room.id
      } as RoomAddRes
    } as ApiResponse)
  } catch (error) {
    console.error('创建房间失败:', error)
    return NextResponse.json({
      message: '创建房间失败',
      code: 1,
      data: null
    } as ApiResponse, { status: 500 })
  }
}
