package main

import (
	"net/http"

	"github.com/sirupsen/logrus"
	"github.com/zyz/comment-api/db" // 保留数据库包导入（正确）
)

// ------------------------------
// CORS中间件（保持不变，解决跨域问题）
// ------------------------------
func corsMiddleware(next http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		// 允许的前端Origin（修改为你的前端实际地址，如http://localhost:5500）
		allowedOrigins := []string{
			"http://localhost:5500", // 保留原有的localhost
			"http://127.0.0.1:5500", // 添加前端实际的Origin（Live Server的地址）
		}
		origin := r.Header.Get("Origin")

		for _, o := range allowedOrigins {
			if o == origin {
				w.Header().Set("Access-Control-Allow-Origin", origin)
				break
			}
		}

		w.Header().Set("Access-Control-Allow-Methods", "GET, POST, OPTIONS")
		w.Header().Set("Access-Control-Allow-Headers", "Content-Type")

		if r.Method == http.MethodOptions {
			w.WriteHeader(http.StatusOK)
			return
		}

		next.ServeHTTP(w, r)
	})
}

func main() {
	// 初始化日志（保持不变）
	logrus.SetFormatter(&logrus.JSONFormatter{})
	logrus.Info("启动服务器...")

	// 初始化数据库（修改：接收error并处理）
	if err := db.InitDB(); err != nil { // 现在db.InitDB()返回error，可正确接收
		logrus.WithError(err).Fatal("数据库初始化失败") // 这里用Fatal退出程序
	}
	logrus.Info("PostgreSQL 数据库初始化成功")

	// 注册路由（直接使用main包中的函数，无需前缀）
	mux := http.NewServeMux()
	mux.HandleFunc("/comment/get", GetCommentsHandler)      // 直接使用GetCommentsHandler（来自server.go）
	mux.HandleFunc("/comment/add", AddCommentHandler)       // 直接使用AddCommentHandler（来自server.go）
	mux.HandleFunc("/comment/delete", DeleteCommentHandler) // 直接使用DeleteCommentHandler（来自server.go）

	// 应用CORS中间件（保持不变）
	handler := corsMiddleware(mux)

	// 启动服务器（保持不变）
	addr := ":8080"
	logrus.WithField("address", addr).Info("服务器启动")
	if err := http.ListenAndServe(addr, handler); err != nil {
		logrus.WithError(err).Fatal("服务器启动失败")
	}
}
