package main // 必须是main包，和main.go同目录

import (
	"encoding/json"
	"net/http"
	"strconv"

	"github.com/sirupsen/logrus"
	"github.com/zyz/comment-api/db"
	"gorm.io/gorm"
)

// Response 统一响应格式（保持不变）
type Response struct {
	Code int         `json:"code"`
	Msg  string      `json:"msg"`
	Data interface{} `json:"data"`
}

// GetCommentsResponse 获取评论响应（保持不变）
type GetCommentsResponse struct {
	Total    int64        `json:"total"`
	Comments []db.Comment `json:"comments"`
}

// ------------------------------
// 导出函数（首字母大写，main包中可直接使用）
// ------------------------------
func GetCommentsHandler(w http.ResponseWriter, r *http.Request) { // 首字母大写，导出
	log := logrus.WithFields(logrus.Fields{
		"method": r.Method,
		"path":   r.URL.Path,
	})

	if r.Method != http.MethodGet {
		respondError(w, http.StatusMethodNotAllowed, "方法不被允许", nil, log)
		return
	}

	page, size := parsePaginationParams(r)
	log = log.WithFields(logrus.Fields{
		"page": page,
		"size": size,
	})

	comments, total, err := db.GetComments(page, size)
	if err != nil {
		respondError(w, http.StatusInternalServerError, "查询评论失败", nil, log.WithError(err))
		return
	}

	data := GetCommentsResponse{
		Total:    total,
		Comments: comments,
	}
	respondSuccess(w, data, log)
}

func AddCommentHandler(w http.ResponseWriter, r *http.Request) { // 首字母大写，导出
	log := logrus.WithFields(logrus.Fields{
		"method": r.Method,
		"path":   r.URL.Path,
	})

	if r.Method != http.MethodPost {
		respondError(w, http.StatusMethodNotAllowed, "方法不被允许", nil, log)
		return
	}

	var req struct {
		Name    string `json:"name"`
		Content string `json:"content"`
	}
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		respondError(w, http.StatusBadRequest, "无效的请求体", nil, log.WithError(err))
		return
	}

	if req.Name == "" || req.Content == "" {
		respondError(w, http.StatusBadRequest, "姓名和内容为必填项", nil, log)
		return
	}

	comment, err := db.CreateComment(req.Name, req.Content)
	if err != nil {
		respondError(w, http.StatusInternalServerError, "创建评论失败", nil, log.WithError(err))
		return
	}

	respondSuccess(w, comment, log)
}

func DeleteCommentHandler(w http.ResponseWriter, r *http.Request) { // 首字母大写，导出
	log := logrus.WithFields(logrus.Fields{
		"method": r.Method,
		"path":   r.URL.Path,
	})

	if r.Method != http.MethodPost {
		respondError(w, http.StatusMethodNotAllowed, "方法不被允许", nil, log)
		return
	}

	idStr := r.URL.Query().Get("id")
	id, err := strconv.Atoi(idStr)
	if err != nil || id <= 0 {
		respondError(w, http.StatusBadRequest, "无效的评论 ID", nil, log.WithError(err))
		return
	}

	if err := db.DeleteComment(id); err != nil {
		if err == gorm.ErrRecordNotFound {
			respondError(w, http.StatusNotFound, "评论不存在", nil, log)
		} else {
			respondError(w, http.StatusInternalServerError, "删除评论失败", nil, log.WithError(err))
		}
		return
	}

	respondSuccess(w, nil, log)
}

// parsePaginationParams 解析分页参数（保持不变）
func parsePaginationParams(r *http.Request) (page, size int) {
	pageStr := r.URL.Query().Get("page")
	sizeStr := r.URL.Query().Get("size")

	page, err := strconv.Atoi(pageStr)
	if err != nil || page < 1 {
		page = 1
	}

	size, err = strconv.Atoi(sizeStr)
	if err != nil || size < -1 || size == 0 {
		size = 10
	}

	return page, size
}

// respondSuccess 发送成功响应（保持不变）
func respondSuccess(w http.ResponseWriter, data interface{}, log *logrus.Entry) {
	resp := Response{
		Code: 0,
		Msg:  "成功",
		Data: data,
	}
	sendResponse(w, http.StatusOK, resp, log)
}

// respondError 发送错误响应（保持不变）
func respondError(w http.ResponseWriter, status int, msg string, data interface{}, log *logrus.Entry) {
	resp := Response{
		Code: status,
		Msg:  msg,
		Data: data,
	}
	log.WithFields(logrus.Fields{
		"status": status,
		"msg":    msg,
	}).Error("请求失败")
	sendResponse(w, status, resp, log)
}

// sendResponse 发送JSON响应（保持不变）
func sendResponse(w http.ResponseWriter, status int, resp Response, log *logrus.Entry) {
	// 关键：添加charset=utf-8
	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	w.WriteHeader(status)
	if err := json.NewEncoder(w).Encode(resp); err != nil {
		log.WithError(err).Error("发送响应失败")
	}
}
