package db

import (
	"github.com/sirupsen/logrus"
	"gorm.io/gorm"
)

// Comment 数据库模型
type Comment struct {
	ID      int    `json:"id" gorm:"primaryKey;autoIncrement"`
	Name    string `json:"name" gorm:"not null"`
	Comment string `json:"content" gorm:"not null"`
}

// CreateComment 创建评论
func CreateComment(name, content string) (*Comment, error) {
	comment := &Comment{
		Name:    name,
		Comment: content,
	}
	if err := DB.Create(comment).Error; err != nil {
		logrus.WithError(err).Error("创建评论失败")
		return nil, err
	}
	logrus.WithFields(logrus.Fields{
		"comment_id": comment.ID,
		"name":       comment.Name,
	}).Info("评论创建成功")
	return comment, nil
}

// GetComments 获取评论（支持分页）
func GetComments(page, size int) ([]Comment, int64, error) {
	var comments []Comment
	var total int64

	// 查询总数
	if err := DB.Model(&Comment{}).Count(&total).Error; err != nil {
		logrus.WithError(err).Error("获取评论总数失败")
		return nil, 0, err
	}

	// 分页查询
	if size == -1 {
		// 返回所有评论
		if err := DB.Find(&comments).Error; err != nil {
			logrus.WithError(err).Error("获取所有评论失败")
			return nil, 0, err
		}
	} else {
		// 分页
		offset := (page - 1) * size
		if err := DB.Limit(size).Offset(offset).Find(&comments).Error; err != nil {
			logrus.WithError(err).Error("获取分页评论失败")
			return nil, 0, err
		}
	}

	logrus.WithFields(logrus.Fields{
		"page":  page,
		"size":  size,
		"total": total,
	}).Info("评论查询成功")
	return comments, total, nil
}

// DeleteComment 删除评论
func DeleteComment(id int) error {
	result := DB.Where("id = ?", id).Delete(&Comment{})
	if err := result.Error; err != nil {
		logrus.WithError(err).Error("删除评论失败")
		return err
	}
	if result.RowsAffected == 0 {
		logrus.WithField("comment_id", id).Warn("评论不存在")
		return gorm.ErrRecordNotFound
	}
	logrus.WithField("comment_id", id).Info("评论删除成功")
	return nil
}
