import prisma from "@/lib/prisma";

export async function GET(request: Request) {
    try {
      const { searchParams } = new URL(request.url);
      const roomId = searchParams.get("roomId");
      
      if (!roomId) {
        return new Response("缺少房间ID", { status: 400 });
      }
      
      const messages = await prisma.message.findMany({
        where: { roomId: Number(roomId) },
        include: { user: true },
        orderBy: { createdAt: "asc" }
      });
      
      // 添加类型注解解决隐式 any 问题
      return Response.json(messages.map((msg: { 
        id: number; 
        content: string; 
        user: { username: string }; 
        createdAt: Date 
      }) => ({
        id: msg.id.toString(),
        content: msg.content,
        sender: msg.user.username,
        timestamp: msg.createdAt
      })));
    } catch (error) {
      console.error("获取消息失败:", error);
      return new Response("服务器错误", { status: 500 });
    }
  }

export async function POST(request: Request) {
  try {
    const { content, roomId, sender } = await request.json();
    
    if (!content || !roomId || !sender) {
      return new Response("缺少必要参数", { status: 400 });
    }
    
    // 查找或创建用户
    let user = await prisma.user.findUnique({ where: { username: sender } });
    if (!user) {
      user = await prisma.user.create({ data: { username: sender } });
    }
    
    // 创建消息
    const message = await prisma.message.create({
      data: {
        content,
        userId: user.id,
        roomId: Number(roomId)
      }
    });
    
    return Response.json({
      id: message.id.toString(),
      content: message.content,
      sender: user.username,
      timestamp: message.createdAt
    });
  } catch (error) {
    console.error("发送消息失败:", error);
    return new Response("服务器错误", { status: 500 });
  }
}