"use client";

import { useState, useEffect } from "react";
import useSWR, { mutate } from "swr";
import { PlusCircleIcon } from "@heroicons/react/24/outline";

type Room = {
  id: string;
  name: string;
};

type Message = {
  id: string;
  content: string;
  sender: string;
  timestamp: Date;
};

// 数据获取函数
const fetcher = (url: string) => fetch(url).then((res) => res.json());

export default function ChatRoom() {
  const [activeRoom, setActiveRoom] = useState("1");
  const [newMessage, setNewMessage] = useState("");
  const [username, setUsername] = useState("Guest");
  const [isCreatingRoom, setIsCreatingRoom] = useState(false);
  const [newRoomName, setNewRoomName] = useState("");

  // 从localStorage获取用户名
  useEffect(() => {
    const storedUsername = localStorage.getItem("chat-username");
    if (storedUsername) {
      setUsername(storedUsername);
    }
  }, []);

  // 使用SWR获取房间列表
  const { data: rooms, error: roomsError, mutate: mutateRooms } = useSWR<Room[]>(
    "/api/rooms",
    fetcher
  );

  // 使用SWR获取当前房间的消息
  const { data: messages, error: messagesError } = useSWR<Message[]>(
    `/api/messages?roomId=${activeRoom}`,
    fetcher,
    { refreshInterval: 3000 } // 每3秒刷新一次
  );

  const handleSendMessage = async () => {
    if (newMessage.trim()) {
      try {
        // 发送消息到后端
        await fetch("/api/messages", {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({
            content: newMessage.trim(),
            roomId: activeRoom,
            sender: username,
          }),
        });

        setNewMessage("");
        // 刷新消息数据
        mutate(`/api/messages?roomId=${activeRoom}`);
      } catch (error) {
        console.error("发送消息失败:", error);
      }
    }
  };

  // 创建新房间的函数
  const handleCreateRoom = async () => {
    if (!newRoomName.trim()) return;
    
    try {
      const response = await fetch("/api/rooms", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ name: newRoomName.trim() })
      });
      
      if (response.ok) {
        // 刷新房间列表
        mutateRooms();
        setNewRoomName("");
        setIsCreatingRoom(false);
      }
    } catch (error) {
      console.error("创建房间失败:", error);
    }
  };

  // 生成用户头像
  const getAvatar = (name: string) => {
    return name.charAt(0).toUpperCase();
  };

  return (
    <div className="flex h-screen">
      {/* 房间列表 */}
      <div className="w-1/4 bg-gray-100 border-r p-4 flex flex-col">
        {/* 标题和创建按钮 */}
        <div className="flex justify-between items-center mb-4">
          <h2 className="text-xl font-bold">聊天房间</h2>
          <button
            onClick={() => setIsCreatingRoom(true)}
            className="p-1 text-blue-500 hover:text-blue-700"
            title="创建新房间"
          >
            <PlusCircleIcon className="h-6 w-6" />
          </button>
        </div>
        
        {/* 创建房间表单 */}
        {isCreatingRoom && (
          <div className="mb-4 p-2 bg-white rounded shadow">
            <input
              type="text"
              value={newRoomName}
              onChange={(e) => setNewRoomName(e.target.value)}
              placeholder="输入房间名称"
              className="w-full px-2 py-1 border rounded mb-2"
              autoFocus
              onKeyPress={(e) => e.key === "Enter" && handleCreateRoom()}
            />
            <div className="flex space-x-2">
              <button
                onClick={handleCreateRoom}
                className="flex-1 bg-blue-500 text-white py-1 rounded hover:bg-blue-600"
              >
                创建
              </button>
              <button
                onClick={() => setIsCreatingRoom(false)}
                className="flex-1 bg-gray-300 text-gray-700 py-1 rounded hover:bg-gray-400"
              >
                取消
              </button>
            </div>
          </div>
        )}
        
        {/* 房间列表内容 */}
        {roomsError ? (
          <div className="text-red-500">加载房间失败</div>
        ) : !rooms ? (
          <div>加载房间中...</div>
        ) : (
          <ul>
            {rooms.map((room: Room) => (
              <li
                key={room.id}
                className={`p-3 mb-2 rounded cursor-pointer ${
                  activeRoom === room.id
                    ? "bg-blue-500 text-white"
                    : "hover:bg-gray-200"
                }`}
                onClick={() => setActiveRoom(room.id)}
              >
                {room.name}
              </li>
            ))}
          </ul>
        )}
      </div>

      {/* 聊天区域 */}
      <div className="flex-1 flex flex-col">
        {/* 消息列表 */}
        <div className="flex-1 p-4 overflow-y-auto">
          {messagesError ? (
            <div className="text-red-500">加载消息失败</div>
          ) : !messages ? (
            <div>加载消息中...</div>
          ) : (
            messages.map((msg: Message) => (
              <div key={msg.id} className="mb-4 flex">
                <div className="w-10 h-10 rounded-full bg-blue-500 flex items-center justify-center text-white font-bold mr-3">
                  {getAvatar(msg.sender)}
                </div>
                <div>
                  <div className="font-bold">
                    {msg.sender}
                    <span className="text-gray-500 text-sm ml-2">
                      {new Date(msg.timestamp).toLocaleTimeString()}
                    </span>
                  </div>
                  <div>{msg.content}</div>
                </div>
              </div>
            ))
          )}
        </div>

        {/* 消息输入框 */}
        <div className="p-4 border-t">
          <div className="flex">
            <input
              type="text"
              value={newMessage}
              onChange={(e) => setNewMessage(e.target.value)}
              placeholder="输入消息..."
              className="flex-1 px-4 py-2 border rounded-l"
              onKeyPress={(e) => e.key === "Enter" && handleSendMessage()}
            />
            <button
              onClick={handleSendMessage}
              className="bg-blue-500 text-white px-4 py-2 rounded-r"
            >
              发送
            </button>
          </div>
        </div>
      </div>
    </div>
  );
}