import { EventEmitter } from 'events';

// 每个房间一个事件总线，避免全局广播
const roomEmitters: Map<number, EventEmitter> = new Map();

function getRoomEmitter(roomId: number): EventEmitter {
  let emitter = roomEmitters.get(roomId);
  if (!emitter) {
    emitter = new EventEmitter();
    // 避免内存泄漏警告（聊天室场景允许更多监听者）
    emitter.setMaxListeners(1000);
    roomEmitters.set(roomId, emitter);
  }
  return emitter;
}

export type SseMessage = {
  type: 'message' | 'system';
  payload: unknown;
};

export function publishToRoom(roomId: number, data: SseMessage) {
  const emitter = getRoomEmitter(roomId);
  emitter.emit('event', data);
}

export function subscribeRoom(roomId: number, onEvent: (data: SseMessage) => void) {
  const emitter = getRoomEmitter(roomId);
  const handler = (data: SseMessage) => onEvent(data);
  emitter.on('event', handler);
  return () => {
    emitter.off('event', handler);
  };
}


