import jwt from 'jsonwebtoken';
import bcrypt from 'bcryptjs';
import { JWTPayload } from '@/types/auth';

// 获取环境变量
const JWT_SECRET = process.env.JWT_SECRET || 'fallback-secret-key';
const JWT_REFRESH_SECRET = process.env.JWT_REFRESH_SECRET || 'fallback-refresh-secret-key';
const JWT_EXPIRES_IN = process.env.JWT_EXPIRES_IN || '15m';
const JWT_REFRESH_EXPIRES_IN = process.env.JWT_REFRESH_EXPIRES_IN || '7d';
const BCRYPT_SALT_ROUNDS = parseInt(process.env.BCRYPT_SALT_ROUNDS || '12');

/**
 * 密码加密
 * @param password 原始密码
 * @returns 加密后的密码哈希
 */
export async function hashPassword(password: string): Promise<string> {
  try {
    const salt = await bcrypt.genSalt(BCRYPT_SALT_ROUNDS);
    const hash = await bcrypt.hash(password, salt);
    return hash;
  } catch (error) {
    console.error('密码加密失败:', error);
    throw new Error('密码加密失败');
  }
}

/**
 * 密码验证
 * @param password 原始密码
 * @param hash 加密后的密码哈希
 * @returns 是否匹配
 */
export async function verifyPassword(password: string, hash: string): Promise<boolean> {
  try {
    return await bcrypt.compare(password, hash);
  } catch (error) {
    console.error('密码验证失败:', error);
    return false;
  }
}

/**
 * 生成访问令牌
 * @param payload JWT载荷
 * @returns 访问令牌
 */
export function generateAccessToken(payload: Omit<JWTPayload, 'iat' | 'exp'>): string {
  try {
    return jwt.sign(payload, JWT_SECRET, {
      expiresIn: JWT_EXPIRES_IN,
      issuer: 'chatroom-app',
      audience: 'chatroom-users'
    } as jwt.SignOptions);
  } catch (error) {
    console.error('生成访问令牌失败:', error);
    throw new Error('生成访问令牌失败');
  }
}

/**
 * 生成刷新令牌
 * @param userId 用户ID
 * @returns 刷新令牌
 */
export function generateRefreshToken(userId: string): string {
  try {
    return jwt.sign(
      { userId, type: 'refresh' },
      JWT_REFRESH_SECRET,
      {
        expiresIn: JWT_REFRESH_EXPIRES_IN,
        issuer: 'chatroom-app',
        audience: 'chatroom-users'
      } as jwt.SignOptions
    );
  } catch (error) {
    console.error('生成刷新令牌失败:', error);
    throw new Error('生成刷新令牌失败');
  }
}

/**
 * 验证访问令牌
 * @param token 访问令牌
 * @returns JWT载荷或null
 */
export function verifyAccessToken(token: string): JWTPayload | null {
  try {
    const decoded = jwt.verify(token, JWT_SECRET, {
      issuer: 'chatroom-app',
      audience: 'chatroom-users'
    }) as JWTPayload;
    
    return decoded;
  } catch (error) {
    if (error instanceof jwt.TokenExpiredError) {
      console.log('访问令牌已过期');
    } else if (error instanceof jwt.JsonWebTokenError) {
      console.log('访问令牌无效');
    } else {
      console.error('验证访问令牌失败:', error);
    }
    return null;
  }
}

/**
 * 验证刷新令牌
 * @param token 刷新令牌
 * @returns 用户ID或null
 */
export function verifyRefreshToken(token: string): string | null {
  try {
    const decoded = jwt.verify(token, JWT_REFRESH_SECRET, {
      issuer: 'chatroom-app',
      audience: 'chatroom-users'
    }) as { userId: string; type: string };
    
    if (decoded.type !== 'refresh') {
      console.log('令牌类型错误');
      return null;
    }
    
    return decoded.userId;
  } catch (error) {
    if (error instanceof jwt.TokenExpiredError) {
      console.log('刷新令牌已过期');
    } else if (error instanceof jwt.JsonWebTokenError) {
      console.log('刷新令牌无效');
    } else {
      console.error('验证刷新令牌失败:', error);
    }
    return null;
  }
}

/**
 * 从请求头中提取令牌
 * @param authorizationHeader Authorization请求头
 * @returns 令牌或null
 */
export function extractTokenFromHeader(authorizationHeader: string | null): string | null {
  if (!authorizationHeader) {
    return null;
  }
  
  const parts = authorizationHeader.split(' ');
  if (parts.length !== 2 || parts[0] !== 'Bearer') {
    return null;
  }
  
  return parts[1];
}

/**
 * 生成随机密码
 * @param length 密码长度
 * @returns 随机密码
 */
export function generateRandomPassword(length: number = 12): string {
  const charset = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!@#$%^&*';
  let password = '';
  
  for (let i = 0; i < length; i++) {
    const randomIndex = Math.floor(Math.random() * charset.length);
    password += charset[randomIndex];
  }
  
  return password;
}
