import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/db';
import { ApiResponse, RoomMessageListRes } from '@/types/chat';

/**
 * 获取房间消息列表 API
 * GET /api/room/message/list?roomId={roomId}
 */
export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const roomId = searchParams.get('roomId');

    // 参数验证
    if (!roomId) {
      return NextResponse.json(
        {
          message: '缺少房间ID参数',
          code: 400,
          data: null
        } as ApiResponse,
        { status: 400 }
      );
    }

    const roomIdNum = parseInt(roomId);
    if (isNaN(roomIdNum)) {
      return NextResponse.json(
        {
          message: '房间ID格式错误',
          code: 400,
          data: null
        } as ApiResponse,
        { status: 400 }
      );
    }

    // 检查房间是否存在
    const room = await prisma.room.findUnique({
      where: { roomId: roomIdNum }
    });

    if (!room) {
      return NextResponse.json(
        {
          message: '房间不存在',
          code: 404,
          data: null
        } as ApiResponse,
        { status: 404 }
      );
    }

    // 获取房间消息列表
    const messages = await prisma.message.findMany({
      where: { roomId: roomIdNum },
      orderBy: { time: 'asc' },
      take: 100 // 限制返回最近100条消息
    });

    const response: RoomMessageListRes = {
      messages: messages.map(msg => ({
        messageId: msg.messageId,
        roomId: msg.roomId,
        sender: msg.sender,
        content: msg.content,
        time: Number(msg.time) // 将 BigInt 转换为 Number
      }))
    };

    return NextResponse.json(
      {
        message: '获取消息列表成功',
        code: 0,
        data: response
      } as ApiResponse<RoomMessageListRes>,
      { status: 200 }
    );

  } catch (error) {
    console.error('获取消息列表失败:', error);
    
    return NextResponse.json(
      {
        message: '服务器内部错误',
        code: 500,
        data: null
      } as ApiResponse,
      { status: 500 }
    );
  }
}
