import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/db';
import { ApiResponse, RoomListRes } from '@/types/chat';

/**
 * 获取房间列表 API
 * GET /api/room/list
 */
export async function GET(request: NextRequest) {
  try {
    // 获取所有房间
    const rooms = await prisma.room.findMany({
      orderBy: { updatedAt: 'desc' }
    });

    // 获取每个房间的最后一条消息
    const roomsWithLastMessage = await Promise.all(
      rooms.map(async (room) => {
        const lastMessage = await prisma.message.findFirst({
          where: { roomId: room.roomId },
          orderBy: { time: 'desc' }
        });

        return {
          roomId: room.roomId,
          roomName: room.roomName,
          lastMessage: lastMessage ? {
            messageId: lastMessage.messageId,
            roomId: lastMessage.roomId,
            sender: lastMessage.sender,
            content: lastMessage.content,
            time: Number(lastMessage.time) // 将 BigInt 转换为 Number
          } : null
        };
      })
    );

    const response: RoomListRes = {
      rooms: roomsWithLastMessage
    };

    return NextResponse.json(
      {
        message: '获取房间列表成功',
        code: 0,
        data: response
      } as ApiResponse<RoomListRes>,
      { status: 200 }
    );

  } catch (error) {
    console.error('获取房间列表失败:', error);
    
    return NextResponse.json(
      {
        message: '服务器内部错误',
        code: 500,
        data: null
      } as ApiResponse,
      { status: 500 }
    );
  }
}
