import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/db';
import { Prisma } from '@prisma/client';
import { hashPassword, generateAccessToken, generateRefreshToken } from '@/lib/auth';
import { RegisterRequest, ApiResponse, AuthResponse, UserStatus } from '@/types/auth';

/**
 * 用户注册 API
 * POST /api/auth/register
 */
export async function POST(request: NextRequest) {
  try {
    // 开发期邮箱白名单（仅用于调试场景）
    const EMAIL_WHITELIST = ['210914403@qq.com'];
    // 解析请求体
    const body: RegisterRequest = await request.json();
    const { username, email, password, nickname } = body;

    // 参数验证
    if (!username || !email || !password || !nickname) {
      return NextResponse.json(
        {
          message: '缺少必要参数',
          code: 400,
          data: null
        } as ApiResponse,
        { status: 400 }
      );
    }

    // 用户名长度验证
    if (username.length < 3 || username.length > 20) {
      return NextResponse.json(
        {
          message: '用户名长度必须在3-20个字符之间',
          code: 400,
          data: null
        } as ApiResponse,
        { status: 400 }
      );
    }

    // 邮箱格式验证
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(email)) {
      return NextResponse.json(
        {
          message: '邮箱格式不正确',
          code: 400,
          data: null
        } as ApiResponse,
        { status: 400 }
      );
    }

    // 密码强度验证
    if (password.length < 6) {
      return NextResponse.json(
        {
          message: '密码长度至少6个字符',
          code: 400,
          data: null
        } as ApiResponse,
        { status: 400 }
      );
    }

    // 昵称长度验证
    if (nickname.length < 2 || nickname.length > 20) {
      return NextResponse.json(
        {
          message: '昵称长度必须在2-20个字符之间',
          code: 400,
          data: null
        } as ApiResponse,
        { status: 400 }
      );
    }

    // 检查用户名是否为admin（不区分大小写）
    if (username.toLowerCase() === 'admin') {
      return NextResponse.json(
        {
          message: 'admin用户名已被保留，请选择其他用户名',
          code: 400,
          data: null
        } as ApiResponse,
        { status: 400 }
      );
    }

    // 检查昵称是否为admin（不区分大小写）
    if (nickname.toLowerCase() === 'admin') {
      return NextResponse.json(
        {
          message: 'admin昵称已被保留，请选择其他昵称',
          code: 400,
          data: null
        } as ApiResponse,
        { status: 400 }
      );
    }

    // 测试账户保留：禁止注册 test/test@qq.com（无论是否已存在）
    if (username.toLowerCase() === 'test') {
      return NextResponse.json(
        {
          message: '用户名已存在',
          code: 409,
          data: null
        } as ApiResponse,
        { status: 409 }
      );
    }

    if (email.toLowerCase() === 'test@qq.com') {
      return NextResponse.json(
        {
          message: '邮箱已被注册',
          code: 409,
          data: null
        } as ApiResponse,
        { status: 409 }
      );
    }

    // 检查用户名/邮箱是否已存在
    const existingUser = await prisma.user.findFirst({
      where: {
        OR: [
          { username },
          { email }
        ]
      }
    });

    if (existingUser) {
      // 用户名冲突
      if (existingUser.username === username) {
        return NextResponse.json(
          {
            message: '用户名已存在',
            code: 409,
            data: null
          } as ApiResponse,
          { status: 409 }
        );
      }

      // 邮箱冲突
      if (existingUser.email === email) {
        // 如果是白名单邮箱，直接签发登录态返回（方便调试）
        if (EMAIL_WHITELIST.includes(email)) {
          const accessTokenForExisting = generateAccessToken({
            userId: existingUser.id,
            username: existingUser.username,
            email: existingUser.email
          });
          const refreshTokenForExisting = generateRefreshToken(existingUser.id);

          const authResponseExisting: AuthResponse = {
            user: {
              id: existingUser.id,
              username: existingUser.username,
              email: existingUser.email,
              nickname: existingUser.nickname,
              avatar: existingUser.avatar || undefined,
              status: existingUser.status as UserStatus,
              createdAt: existingUser.createdAt,
              updatedAt: existingUser.updatedAt
            },
            accessToken: accessTokenForExisting,
            refreshToken: refreshTokenForExisting
          };

          return NextResponse.json(
            {
              message: '白名单邮箱已存在，已为您自动登录',
              code: 0,
              data: authResponseExisting
            } as ApiResponse<AuthResponse>,
            { status: 200 }
          );
        }

        return NextResponse.json(
          {
            message: '邮箱已被注册',
            code: 409,
            data: null
          } as ApiResponse,
          { status: 409 }
        );
      }
    }

    // 加密密码
    const hashedPassword = await hashPassword(password);

    // 创建用户
    const user = await prisma.user.create({
      data: {
        username,
        email,
        password: hashedPassword,
        nickname,
        status: 'OFFLINE'
      }
    });

    // 生成令牌
    const accessToken = generateAccessToken({
      userId: user.id,
      username: user.username,
      email: user.email
    });

    const refreshToken = generateRefreshToken(user.id);

    // 构建响应数据
    const authResponse: AuthResponse = {
      user: {
        id: user.id,
        username: user.username,
        email: user.email,
        nickname: user.nickname,
        avatar: user.avatar || undefined,
        status: user.status as UserStatus,
        createdAt: user.createdAt,
        updatedAt: user.updatedAt
      },
      accessToken,
      refreshToken
    };

    console.log(`✅ 用户注册成功: ${username} (${user.id})`);

    return NextResponse.json(
      {
        message: '用户注册成功',
        code: 0,
        data: authResponse
      } as ApiResponse<AuthResponse>,
      { status: 201 }
    );

  } catch (error) {
    console.error('用户注册失败:', error);

    // 捕获数据库唯一约束错误，确保重复检测来自DB
    if (error instanceof Prisma.PrismaClientKnownRequestError && error.code === 'P2002') {
      const target = (error.meta?.target as string[]) || [];
      // 根据冲突字段返回精确提示
      if (target.includes('username') || target.join(',').includes('username')) {
        return NextResponse.json(
          { message: '用户名已存在', code: 409, data: null } as ApiResponse,
          { status: 409 }
        );
      }
      if (target.includes('email') || target.join(',').includes('email')) {
        return NextResponse.json(
          { message: '邮箱已被注册', code: 409, data: null } as ApiResponse,
          { status: 409 }
        );
      }
      // 未知字段冲突，通用重复提示
      return NextResponse.json(
        { message: '信息已存在', code: 409, data: null } as ApiResponse,
        { status: 409 }
      );
    }

    return NextResponse.json(
      {
        message: '服务器内部错误',
        code: 500,
        data: null
      } as ApiResponse,
      { status: 500 }
    );
  }
}
