import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/db';
import { RoomDeleteArgs, ApiResponse } from '@/types/chat';

/**
 * 删除房间 API
 * POST /api/room/delete
 */
export async function POST(request: NextRequest) {
  try {
    // 解析请求体
    const body: RoomDeleteArgs = await request.json();
    const { user, roomId } = body;

    // 参数验证
    if (!user || !roomId) {
      return NextResponse.json(
        {
          message: '缺少必要参数',
          code: 400,
          data: null
        } as ApiResponse,
        { status: 400 }
      );
    }

    // 检查房间是否存在
    const existingRoom = await prisma.room.findUnique({
      where: { roomId }
    });

    if (!existingRoom) {
      return NextResponse.json(
        {
          message: '房间不存在',
          code: 404,
          data: null
        } as ApiResponse,
        { status: 404 }
      );
    }

    // 检查用户权限（只有创建者或管理员可以删除房间）
    const isCreator = existingRoom.creator === user;
    const isAdmin = user.toLowerCase() === 'admin';
    
    if (!isCreator && !isAdmin) {
      return NextResponse.json(
        {
          message: '只有房间创建者或管理员可以删除房间',
          code: 403,
          data: null
        } as ApiResponse,
        { status: 403 }
      );
    }

    // 删除房间（由于设置了级联删除，相关消息也会被删除）
    await prisma.room.delete({
      where: { roomId }
    });

    console.log(`✅ 房间删除成功: ${existingRoom.roomName} (${roomId})`);

    return NextResponse.json(
      {
        message: '房间删除成功',
        code: 0,
        data: null
      } as ApiResponse,
      { status: 200 }
    );

  } catch (error) {
    console.error('房间删除失败:', error);
    
    return NextResponse.json(
      {
        message: '服务器内部错误',
        code: 500,
        data: null
      } as ApiResponse,
      { status: 500 }
    );
  }
}
