import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/db';
import { hashPassword, generateAccessToken, generateRefreshToken } from '@/lib/auth';
import { ApiResponse, AuthResponse, UserStatus } from '@/types/auth';

/**
 * 开发用：创建测试账户
 * POST /api/auth/seed-test-user
 */
export async function POST(_request: NextRequest) {
  try {
    if (process.env.ALLOW_TEST_SEED !== 'true') {
      return NextResponse.json({ message: 'Not Found', code: 404, data: null } as ApiResponse, { status: 404 });
    }
    const username = 'test';
    const email = 'test@qq.com';
    const nickname = 'test';
    const passwordPlain = 'test000';

    const existing = await prisma.user.findFirst({ where: { OR: [{ username }, { email }] } });

    let user = existing;
    if (!existing) {
      const hashedPassword = await hashPassword(passwordPlain);
      user = await prisma.user.create({
        data: {
          username,
          email,
          password: hashedPassword,
          nickname,
          status: 'OFFLINE',
        },
      });
      console.log(`✅ 已创建测试用户: ${username} (${user.id})`);
    } else {
      console.log(`ℹ️ 测试用户已存在: ${username} (${existing.id})`);
    }

    if (!user) {
      return NextResponse.json(
        { message: '创建测试用户失败', code: 500, data: null } as ApiResponse,
        { status: 500 }
      );
    }

    const accessToken = generateAccessToken({ userId: user.id, username: user.username, email: user.email });
    const refreshToken = generateRefreshToken(user.id);

    const authResponse: AuthResponse = {
      user: {
        id: user.id,
        username: user.username,
        email: user.email,
        nickname: user.nickname,
        avatar: user.avatar || undefined,
        status: user.status as UserStatus,
        createdAt: user.createdAt,
        updatedAt: user.updatedAt,
      },
      accessToken,
      refreshToken,
    };

    return NextResponse.json(
      { message: '测试用户就绪', code: 0, data: authResponse } as ApiResponse<AuthResponse>,
      { status: 200 }
    );
  } catch (error) {
    console.error('创建测试用户失败:', error);
    return NextResponse.json(
      { message: '服务器内部错误', code: 500, data: null } as ApiResponse,
      { status: 500 }
    );
  }
}


