import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/db';
import { verifyPassword, generateAccessToken, generateRefreshToken } from '@/lib/auth';
import { LoginRequest, ApiResponse, AuthResponse, UserStatus } from '@/types/auth';

/**
 * 用户登录 API
 * POST /api/auth/login
 */
export async function POST(request: NextRequest) {
  try {
    // 解析请求体
    const body: LoginRequest = await request.json();
    const { username, password } = body;

    // 参数验证
    if (!username || !password) {
      return NextResponse.json(
        {
          message: '用户名和密码不能为空',
          code: 400,
          data: null
        } as ApiResponse,
        { status: 400 }
      );
    }

    // 查找用户
    const user = await prisma.user.findFirst({
      where: {
        OR: [
          { username },
          { email: username } // 支持邮箱登录
        ]
      }
    });

    if (!user) {
      return NextResponse.json(
        {
          message: '用户名或密码错误',
          code: 401,
          data: null
        } as ApiResponse,
        { status: 401 }
      );
    }

    // 验证密码
    const isPasswordValid = await verifyPassword(password, user.password);
    if (!isPasswordValid) {
      return NextResponse.json(
        {
          message: '用户名或密码错误',
          code: 401,
          data: null
        } as ApiResponse,
        { status: 401 }
      );
    }

    // 更新用户状态为在线
    await prisma.user.update({
      where: { id: user.id },
      data: { status: 'ONLINE' }
    });

    // 生成令牌
    const accessToken = generateAccessToken({
      userId: user.id,
      username: user.username,
      email: user.email
    });

    const refreshToken = generateRefreshToken(user.id);

    // 构建响应数据
    const authResponse: AuthResponse = {
      user: {
        id: user.id,
        username: user.username,
        email: user.email,
        nickname: user.nickname,
        avatar: user.avatar || undefined,
        status: UserStatus.ONLINE,
        createdAt: user.createdAt,
        updatedAt: user.updatedAt
      },
      accessToken,
      refreshToken
    };

    console.log(`✅ 用户登录成功: ${user.username} (${user.id})`);

    return NextResponse.json(
      {
        message: '登录成功',
        code: 0,
        data: authResponse
      } as ApiResponse<AuthResponse>,
      { status: 200 }
    );

  } catch (error) {
    console.error('用户登录失败:', error);
    
    return NextResponse.json(
      {
        message: '服务器内部错误',
        code: 500,
        data: null
      } as ApiResponse,
      { status: 500 }
    );
  }
}
