'use client';

import React, { useState } from 'react';
import { Input, Button, App } from 'antd';
import { SendOutlined } from '@ant-design/icons';
import styles from './MessageInput.module.css';

interface MessageInputProps {
  onSendMessage: (message: string) => Promise<void>;
}

export default function MessageInput({ onSendMessage }: MessageInputProps) {
  const [message, setMessage] = useState('');
  const [isSending, setIsSending] = useState(false);
  const { message: messageApi } = App.useApp();

  const handleSend = async () => {
    if (!message.trim() || isSending) return;

    setIsSending(true);
    try {
      await onSendMessage(message.trim());
      setMessage('');
    } catch (error) {
      console.error('发送消息失败:', error);
      messageApi.error('发送消息失败，请稍后重试');
    } finally {
      setIsSending(false);
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      handleSend();
    }
  };

  return (
    <div className={styles.inputContainer}>
      <Input.TextArea
        value={message}
        onChange={(e) => setMessage(e.target.value)}
        onKeyPress={handleKeyPress}
        placeholder="输入消息内容... (Shift + Enter 换行)"
        autoSize={{ minRows: 1, maxRows: 4 }}
        disabled={isSending}
        className={styles.textArea}
      />
      <Button 
        type="primary" 
        onClick={handleSend} 
        loading={isSending}
        icon={<SendOutlined />}
        className={styles.sendButton}
        disabled={!message.trim()}
      >
        发送
      </Button>
    </div>
  );
}
