import { NextRequest } from 'next/server';
import { subscribeRoom } from '@/lib/sse';

export async function GET(request: NextRequest) {
  const { searchParams } = new URL(request.url);
  const roomIdParam = searchParams.get('roomId');
  const roomId = roomIdParam ? Number(roomIdParam) : NaN;

  if (!roomIdParam || Number.isNaN(roomId)) {
    return new Response('Bad Request: roomId required', { status: 400 });
  }

  const stream = new ReadableStream({
    start(controller) {
      const encoder = new TextEncoder();

      // 发送初始注释，防止部分代理超时
      controller.enqueue(encoder.encode(`: connected\n\n`));

      const unsubscribe = subscribeRoom(roomId, (data) => {
        const chunk = `data: ${JSON.stringify(data)}\n\n`;
        controller.enqueue(encoder.encode(chunk));
      });

      // 心跳，避免连接被中断
      const heartbeat = setInterval(() => {
        controller.enqueue(encoder.encode(`: keep-alive ${Date.now()}\n\n`));
      }, 25000);

      // 连接关闭时清理
      const close = () => {
        clearInterval(heartbeat);
        unsubscribe();
        controller.close();
      };

      // @ts-ignore
      request.signal.addEventListener('abort', close);
    },
  });

  return new Response(stream, {
    headers: {
      'Content-Type': 'text/event-stream; charset=utf-8',
      // 明确禁用缓存与中间转换，提示代理不缓冲
      'Cache-Control': 'no-cache, no-transform',
      Connection: 'keep-alive',
      'X-Accel-Buffering': 'no',
      // CORS 允许同源与本地调试（如有需要）
      'Access-Control-Allow-Origin': '*',
      'Access-Control-Allow-Headers': 'Cache-Control, Last-Event-ID, Content-Type',
    },
  });
}


