import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/db';
import { RoomAddArgs, ApiResponse, RoomAddRes } from '@/types/chat';

/**
 * 创建房间 API
 * POST /api/room/add
 */
export async function POST(request: NextRequest) {
  try {
    // 解析请求体
    const body: RoomAddArgs = await request.json();
    const { user, roomName } = body;

    // 参数验证
    if (!user || !roomName) {
      return NextResponse.json(
        {
          message: '缺少必要参数',
          code: 400,
          data: null
        } as ApiResponse,
        { status: 400 }
      );
    }

    // 房间名称长度验证
    if (roomName.length < 2 || roomName.length > 50) {
      return NextResponse.json(
        {
          message: '房间名称长度必须在2-50个字符之间',
          code: 400,
          data: null
        } as ApiResponse,
        { status: 400 }
      );
    }

    // 检查房间名称是否已存在
    const existingRoom = await prisma.room.findFirst({
      where: { roomName }
    });

    if (existingRoom) {
      return NextResponse.json(
        {
          message: '房间名称已存在',
          code: 409,
          data: null
        } as ApiResponse,
        { status: 409 }
      );
    }

    // 创建房间
    const room = await prisma.room.create({
      data: {
        roomName,
        creator: user
      }
    });

    console.log(`✅ 房间创建成功: ${roomName} (${room.roomId})`);

    const response: RoomAddRes = {
      roomId: room.roomId
    };

    return NextResponse.json(
      {
        message: '房间创建成功',
        code: 0,
        data: response
      } as ApiResponse<RoomAddRes>,
      { status: 201 }
    );

  } catch (error) {
    console.error('房间创建失败:', error);
    
    return NextResponse.json(
      {
        message: '服务器内部错误',
        code: 500,
        data: null
      } as ApiResponse,
      { status: 500 }
    );
  }
}
