import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/db';
import { publishToRoom } from '@/lib/sse';
import { MessageAddArgs, ApiResponse } from '@/types/chat';

/**
 * 添加消息 API
 * POST /api/message/add
 */
export async function POST(request: NextRequest) {
  try {
    // 解析请求体
    const body: MessageAddArgs = await request.json();
    const { roomId, content, sender } = body;

    // 参数验证
    if (!roomId || !content || !sender) {
      return NextResponse.json(
        {
          message: '缺少必要参数',
          code: 400,
          data: null
        } as ApiResponse,
        { status: 400 }
      );
    }

    // 检查房间是否存在
    const room = await prisma.room.findUnique({
      where: { roomId }
    });

    if (!room) {
      return NextResponse.json(
        {
          message: '房间不存在',
          code: 404,
          data: null
        } as ApiResponse,
        { status: 404 }
      );
    }

    // 消息内容长度验证
    if (content.length === 0 || content.length > 500) {
      return NextResponse.json(
        {
          message: '消息内容长度必须在1-500个字符之间',
          code: 400,
          data: null
        } as ApiResponse,
        { status: 400 }
      );
    }

    // 创建消息
    const message = await prisma.message.create({
      data: {
        roomId,
        sender,
        content,
        time: BigInt(Date.now())
      }
    });

    // 更新房间的更新时间
    await prisma.room.update({
      where: { roomId },
      data: { updatedAt: new Date() }
    });

    // 推送SSE给订阅者
    publishToRoom(roomId, {
      type: 'message',
      payload: {
        messageId: message.messageId,
        roomId: message.roomId,
        sender: message.sender,
        content: message.content,
        time: Number(message.time),
        createdAt: message.createdAt,
      },
    });

    console.log(`✅ 消息发送成功: ${sender} -> ${room.roomName} (${message.messageId})`);

    return NextResponse.json(
      {
        message: '消息发送成功',
        code: 0,
        data: null
      } as ApiResponse,
      { status: 201 }
    );

  } catch (error) {
    console.error('消息发送失败:', error);
    
    return NextResponse.json(
      {
        message: '服务器内部错误',
        code: 500,
        data: null
      } as ApiResponse,
      { status: 500 }
    );
  }
}
