import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/db';
import { hashPassword, generateAccessToken, generateRefreshToken } from '@/lib/auth';
import { ApiResponse, AuthResponse, UserStatus } from '@/types/auth';

/**
 * 开发/运维用：内置管理员账户
 * 账号信息（按需求）：
 * 用户名：soilchalk；邮箱：210914403@qq.com；昵称：admin；密码：Xzr@210914403
 * POST /api/auth/seed-admin
 */
export async function POST(_request: NextRequest) {
  try {
    // 环境开关：默认关闭，避免误暴露
    if (process.env.ALLOW_ADMIN_SEED !== 'true') {
      return NextResponse.json({ message: 'Not Found', code: 404, data: null } as ApiResponse, { status: 404 });
    }

    // 从环境变量读取管理员信息，避免硬编码凭据
    const username = process.env.ADMIN_USERNAME || '';
    const email = process.env.ADMIN_EMAIL || '';
    const nickname = process.env.ADMIN_NICKNAME || 'admin';
    const passwordPlain = process.env.ADMIN_PASSWORD || '';

    if (!username || !email || !passwordPlain) {
      return NextResponse.json(
        { message: '管理员种子变量未配置', code: 403, data: null } as ApiResponse,
        { status: 403 }
      );
    }

    // 如果已存在则复用
    let user = await prisma.user.findFirst({ where: { OR: [{ username }, { email }] } });

    if (!user) {
      const hashedPassword = await hashPassword(passwordPlain);
      user = await prisma.user.create({
        data: {
          username,
          email,
          password: hashedPassword,
          nickname,
          status: 'OFFLINE',
        },
      });
      console.log(`✅ 已创建内置管理员: ${username} (${user.id})`);
    } else {
      // 确保昵称为 admin
      if (user.nickname !== nickname) {
        user = await prisma.user.update({ where: { id: user.id }, data: { nickname } });
      }
      console.log(`ℹ️ 管理员已存在: ${username} (${user.id})`);
    }

    const accessToken = generateAccessToken({ userId: user.id, username: user.username, email: user.email });
    const refreshToken = generateRefreshToken(user.id);

    const authResponse: AuthResponse = {
      user: {
        id: user.id,
        username: user.username,
        email: user.email,
        nickname: user.nickname,
        avatar: user.avatar || undefined,
        status: user.status as UserStatus,
        createdAt: user.createdAt,
        updatedAt: user.updatedAt,
      },
      accessToken,
      refreshToken,
    };

    return NextResponse.json(
      { message: '管理员账户就绪', code: 0, data: authResponse } as ApiResponse<AuthResponse>,
      { status: 200 }
    );
  } catch (error) {
    console.error('创建管理员账户失败:', error);
    return NextResponse.json(
      { message: '服务器内部错误', code: 500, data: null } as ApiResponse,
      { status: 500 }
    );
  }
}


