import { PrismaClient, UserRole } from '@prisma/client';
import bcrypt from 'bcryptjs';

const prisma = new PrismaClient();

async function main() {
  console.log('🌱 开始创建种子数据...');

  // 创建管理员账户
  const adminPassword = await bcrypt.hash('admin123', 12);
  const admin = await prisma.user.upsert({
    where: { username: 'admin' },
    update: {},
    create: {
      username: 'admin',
      email: 'admin@chatroom.com',
      password: adminPassword,
      nickname: '系统管理员',
      role: UserRole.ADMIN,
      status: 'OFFLINE',
    },
  });

  // 创建测试用户
  const testUsers = [
    {
      username: 'alice',
      email: 'alice@example.com',
      password: 'alice123',
      nickname: '爱丽丝',
      role: UserRole.USER,
    },
    {
      username: 'bob',
      email: 'bob@example.com',
      password: 'bob123',
      nickname: '鲍勃',
      role: UserRole.USER,
    },
    {
      username: 'charlie',
      email: 'charlie@example.com',
      password: 'charlie123',
      nickname: '查理',
      role: UserRole.USER,
    },
    {
      username: 'diana',
      email: 'diana@example.com',
      password: 'diana123',
      nickname: '黛安娜',
      role: UserRole.USER,
    },
    {
      username: 'eve',
      email: 'eve@example.com',
      password: 'eve123',
      nickname: '夏娃',
      role: UserRole.USER,
    },
  ];

  for (const userData of testUsers) {
    const hashedPassword = await bcrypt.hash(userData.password, 12);
    await prisma.user.upsert({
      where: { username: userData.username },
      update: {},
      create: {
        username: userData.username,
        email: userData.email,
        password: hashedPassword,
        nickname: userData.nickname,
        role: userData.role,
        status: 'OFFLINE',
      },
    });
  }

  // 创建测试房间
  const testRooms = [
    {
      roomName: '欢迎大厅',
      creator: 'admin',
    },
    {
      roomName: '技术讨论',
      creator: 'alice',
    },
    {
      roomName: '闲聊天地',
      creator: 'bob',
    },
    {
      roomName: '游戏交流',
      creator: 'charlie',
    },
  ];

  for (const roomData of testRooms) {
    const existingRoom = await prisma.room.findFirst({
      where: { roomName: roomData.roomName },
    });
    
    if (!existingRoom) {
      await prisma.room.create({
        data: {
          roomName: roomData.roomName,
          creator: roomData.creator,
        },
      });
    }
  }

  // 创建一些测试消息
  const welcomeRoom = await prisma.room.findFirst({
    where: { roomName: '欢迎大厅' },
  });

  if (welcomeRoom) {
    const welcomeMessages = [
      {
        roomId: welcomeRoom.roomId,
        sender: 'admin',
        content: '欢迎来到聊天室！',
        time: BigInt(Date.now() - 3600000), // 1小时前
      },
      {
        roomId: welcomeRoom.roomId,
        sender: 'alice',
        content: '大家好，我是爱丽丝！',
        time: BigInt(Date.now() - 3000000), // 50分钟前
      },
      {
        roomId: welcomeRoom.roomId,
        sender: 'bob',
        content: '很高兴认识大家！',
        time: BigInt(Date.now() - 2400000), // 40分钟前
      },
    ];

    for (const messageData of welcomeMessages) {
      await prisma.message.create({
        data: messageData,
      });
    }
  }

  console.log('✅ 种子数据创建完成！');
  console.log('\n📋 测试账户列表:');
  console.log('┌─────────────┬─────────────┬─────────────┬─────────────┐');
  console.log('│   用户名    │    密码     │    昵称     │    角色     │');
  console.log('├─────────────┼─────────────┼─────────────┼─────────────┤');
  console.log('│   admin     │  admin123   │ 系统管理员  │   管理员    │');
  console.log('│   alice     │  alice123   │   爱丽丝    │   普通用户  │');
  console.log('│    bob      │   bob123    │    鲍勃     │   普通用户  │');
  console.log('│  charlie    │ charlie123  │    查理     │   普通用户  │');
  console.log('│   diana     │  diana123   │   黛安娜    │   普通用户  │');
  console.log('│    eve      │   eve123    │    夏娃     │   普通用户  │');
  console.log('└─────────────┴─────────────┴─────────────┴─────────────┘');
  
  console.log('\n🏠 测试房间列表:');
  console.log('• 欢迎大厅 (管理员创建)');
  console.log('• 技术讨论 (爱丽丝创建)');
  console.log('• 闲聊天地 (鲍勃创建)');
  console.log('• 游戏交流 (查理创建)');
  
  console.log('\n💡 使用说明:');
  console.log('1. 管理员账户可以删除任何房间和消息');
  console.log('2. 普通用户只能删除自己创建的房间');
  console.log('3. 所有用户都可以发送消息');
  console.log('4. 使用上述账户登录测试功能');
}

main()
  .catch((e) => {
    console.error('❌ 种子数据创建失败:', e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
