'use client';

import React, { useState, useEffect } from 'react';
import useSWR, { mutate } from 'swr';
import { List, Button, Spin, Alert, Modal, Input, App } from 'antd';
import { PlusOutlined, DeleteOutlined } from '@ant-design/icons';
import { fetcher } from '../lib/fetcher';
import { RoomPreviewInfo, RoomAddArgs, RoomDeleteArgs } from '../types';
import styles from './RoomList.module.css';

interface RoomListProps {
  onSelectRoom: (roomId: number) => void;
  currentRoomId: number | null;
  nickname: string;
}

export default function RoomList({ onSelectRoom, currentRoomId, nickname }: RoomListProps) {
  const [isCreateModalVisible, setIsCreateModalVisible] = useState(false);
  const [newRoomName, setNewRoomName] = useState('');
  const [isCreating, setIsCreating] = useState(false);
  const [isClient, setIsClient] = useState(false);
  const { message, modal } = App.useApp();

  // 客户端检测
  useEffect(() => {
    setIsClient(true);
  }, []);

  const { data, error, isLoading } = useSWR<{ rooms: RoomPreviewInfo[] }>('/api/room/list', fetcher, {
    refreshInterval: 1000, // 每秒轮询一次
  });
  const rooms = data?.rooms || [];

  const handleCreateRoom = async () => {
    if (!newRoomName.trim()) {
      message.error('房间名称不能为空');
      return;
    }

    setIsCreating(true);
    try {
      const args: RoomAddArgs = {
        user: nickname,
        roomName: newRoomName.trim(),
      };

      const response = await fetcher('/api/room/add', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(args),
      });

      message.success('房间创建成功');
      setIsCreateModalVisible(false);
      setNewRoomName('');
      
      // 刷新房间列表
      mutate('/api/room/list');
      
      // 自动进入新创建的房间
      if (response.roomId) {
        onSelectRoom(response.roomId);
      }
    } catch (error) {
      console.error('创建房间失败:', error);
      message.error('创建房间失败');
    } finally {
      setIsCreating(false);
    }
  };

  const handleDeleteRoom = async (roomId: number, event: React.MouseEvent) => {
    event.stopPropagation(); // 防止触发房间选择

    modal.confirm({
      title: '确认删除',
      content: '确定要删除这个房间吗？删除后无法恢复。',
      okText: '确认',
      cancelText: '取消',
      onOk: async () => {
        try {
          const args: RoomDeleteArgs = {
            user: nickname,
            roomId,
          };

          await fetcher('/api/room/delete', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(args),
          });

          message.success('房间删除成功');
          
          // 如果删除的是当前房间，清空选择
          if (currentRoomId === roomId) {
            onSelectRoom(0);
          }
          
          // 刷新房间列表
          mutate('/api/room/list');
        } catch (error) {
          console.error('删除房间失败:', error);
          message.error('删除房间失败');
        }
      },
    });
  };

  if (isLoading) {
    return <div className={styles.centered}><Spin size="large" /></div>;
  }

  if (error) {
    return <Alert message="错误" description="加载房间列表失败" type="error" showIcon />;
  }

  return (
    <div className={styles.roomListContainer}>
      <div className={styles.header}>
        <h2>聊天房间</h2>
        <Button
          type="primary"
          shape="circle"
          icon={<PlusOutlined />}
          onClick={() => setIsCreateModalVisible(true)}
        />
      </div>
      
      <List
        className={styles.list}
        dataSource={rooms}
        renderItem={(room) => (
          <List.Item
            className={`${styles.roomItem} ${currentRoomId === room.roomId ? styles.selectedRoom : ''}`}
            onClick={() => onSelectRoom(room.roomId)}
            actions={[
              <Button
                type="text"
                danger
                icon={<DeleteOutlined />}
                onClick={(e) => handleDeleteRoom(room.roomId, e)}
              />,
            ]}
          >
            <List.Item.Meta 
              title={room.roomName}
              description={
                room.lastMessage 
                  ? `${room.lastMessage.sender}: ${room.lastMessage.content.substring(0, 20)}${room.lastMessage.content.length > 20 ? '...' : ''}`
                  : '暂无消息'
              }
            />
          </List.Item>
        )}
      />

      <Modal
        title="创建新房间"
        open={isCreateModalVisible}
        onOk={handleCreateRoom}
        onCancel={() => {
          setIsCreateModalVisible(false);
          setNewRoomName('');
        }}
        confirmLoading={isCreating}
        okText="创建"
        cancelText="取消"
      >
        <Input
          placeholder="请输入房间名称"
          value={newRoomName}
          onChange={(e) => setNewRoomName(e.target.value)}
          onPressEnter={handleCreateRoom}
          autoFocus
        />
      </Modal>
    </div>
  );
}
