'use client';

import React, { useState } from 'react';
import { Form, Input, Button, Card, message, Space } from 'antd';
import { UserOutlined, LockOutlined, LoginOutlined } from '@ant-design/icons';
import { useRouter } from 'next/navigation';
import { fetcher } from '../../lib/fetcher';
import { LoginRequest } from '../../types/auth';

export default function LoginPage() {
  const [loading, setLoading] = useState(false);
  const [usernameError, setUsernameError] = useState<string>('');
  const [passwordError, setPasswordError] = useState<string>('');
  const router = useRouter();

  const handleLogin = async (values: LoginRequest) => {
    // 清空旧错误
    setUsernameError('');
    setPasswordError('');
    setLoading(true);
    try {
      const response = await fetcher('/api/auth/login', {
        method: 'POST',
        body: JSON.stringify(values),
      });

      message.success('登录成功！');
      
      // 保存用户信息到localStorage
      localStorage.setItem('user', JSON.stringify({
        id: response.user.id,
        username: response.user.username,
        nickname: response.user.nickname,
        email: response.user.email,
        token: response.accessToken,
        refreshToken: response.refreshToken
      }));

      // 跳转到聊天页面
      router.push('/chat');
    } catch (error) {
      console.error('登录失败:', error);
      if (error instanceof Error) {
        if (error.message.includes('用户名或密码错误')) {
          setPasswordError('用户名或密码错误，请检查后重试');
        } else if (error.message.includes('用户不存在')) {
          setUsernameError('用户不存在，请先注册');
        } else {
          message.error(error.message || '登录失败，请检查用户名和密码');
        }
      } else {
        message.error('登录失败，请检查用户名和密码');
      }
    } finally {
      setLoading(false);
    }
  };

  const handleRegister = () => {
    router.push('/register');
  };

  return (
    <div className="font-sans flex items-center justify-center min-h-screen p-8 bg-gray-50">
      <main className="w-full max-w-md">
        <div className="text-center mb-8">
          <h1 className="text-3xl font-bold text-gray-800 mb-2">用户登录</h1>
          <p className="text-gray-600">欢迎回到聊天室</p>
        </div>

        <Card className="shadow-lg">
          <Form
            name="login"
            onFinish={handleLogin}
            autoComplete="off"
            layout="vertical"
            size="large"
          >
            <Form.Item
              name="username"
              label="用户名"
              rules={[
                { required: true, message: '请输入用户名' }
              ]}
            >
              <Input 
                prefix={<UserOutlined />} 
                placeholder="请输入用户名或邮箱"
                status={usernameError ? 'error' : undefined}
                onChange={() => setUsernameError('')}
              />
            </Form.Item>
            {usernameError && (
              <div className="text-red-500 text-xs mt-1 mb-4">{usernameError}</div>
            )}

            <Form.Item
              name="password"
              label="密码"
              rules={[
                { required: true, message: '请输入密码' }
              ]}
            >
              <Input.Password 
                prefix={<LockOutlined />} 
                placeholder="请输入密码"
                status={passwordError ? 'error' : undefined}
                onChange={() => setPasswordError('')}
              />
            </Form.Item>
            {passwordError && (
              <div className="text-red-500 text-xs mt-1 mb-4">{passwordError}</div>
            )}

            <Form.Item>
              <Button 
                type="primary" 
                htmlType="submit" 
                loading={loading}
                icon={<LoginOutlined />}
                className="w-full h-12"
              >
                登录
              </Button>
            </Form.Item>

            <div className="text-center">
              <Space>
                <span className="text-gray-600">还没有账户？</span>
                <Button type="link" onClick={handleRegister} className="p-0">
                  立即注册
                </Button>
              </Space>
            </div>
          </Form>
        </Card>
      </main>
    </div>
  );
}
