#!/usr/bin/env tsx

/**
 * 管理员功能测试脚本
 * 用于验证内置管理员账户和权限功能
 */

import { PrismaClient, UserRole } from '@prisma/client';
import bcrypt from 'bcryptjs';

const prisma = new PrismaClient();

async function testAdminFeatures() {
  console.log('🔐 测试管理员功能...\n');

  try {
    // 1. 验证管理员账户存在
    console.log('1️⃣ 验证管理员账户...');
    const admin = await prisma.user.findUnique({
      where: { username: 'admin' },
    });

    if (!admin) {
      console.log('❌ 管理员账户不存在，请先运行: pnpm db:seed');
      return;
    }

    console.log('✅ 管理员账户存在');
    console.log(`   用户名: ${admin.username}`);
    console.log(`   昵称: ${admin.nickname}`);
    console.log(`   角色: ${admin.role}`);
    console.log(`   邮箱: ${admin.email}`);

    // 2. 验证密码
    console.log('\n2️⃣ 验证管理员密码...');
    const isPasswordValid = await bcrypt.compare('admin123', admin.password);
    console.log(isPasswordValid ? '✅ 密码验证通过' : '❌ 密码验证失败');

    // 3. 显示所有测试用户
    console.log('\n3️⃣ 显示所有测试用户...');
    const allUsers = await prisma.user.findMany({
      select: {
        username: true,
        nickname: true,
        email: true,
        role: true,
        status: true,
        createdAt: true,
      },
      orderBy: { createdAt: 'asc' },
    });

    console.log('┌─────────────┬─────────────┬─────────────────────────┬─────────────┬─────────────┐');
    console.log('│   用户名    │    昵称     │          邮箱           │    角色     │    状态     │');
    console.log('├─────────────┼─────────────┼─────────────────────────┼─────────────┼─────────────┤');
    
    allUsers.forEach(user => {
      const roleIcon = user.role === UserRole.ADMIN ? '👑' : '👤';
      const statusIcon = user.status === 'ONLINE' ? '🟢' : '⚪';
      console.log(`│ ${user.username.padEnd(11)} │ ${user.nickname.padEnd(11)} │ ${user.email.padEnd(23)} │ ${roleIcon} ${user.role.padEnd(9)} │ ${statusIcon} ${user.status.padEnd(9)} │`);
    });
    
    console.log('└─────────────┴─────────────┴─────────────────────────┴─────────────┴─────────────┘');

    // 4. 显示所有房间
    console.log('\n4️⃣ 显示所有房间...');
    const allRooms = await prisma.room.findMany({
      include: {
        messages: {
          take: 1,
          orderBy: { createdAt: 'desc' },
        },
      },
      orderBy: { createdAt: 'asc' },
    });

    console.log('┌─────────────┬─────────────┬─────────────┬─────────────┐');
    console.log('│   房间ID    │   房间名称  │    创建者   │  最后消息   │');
    console.log('├─────────────┼─────────────┼─────────────┼─────────────┤');
    
    allRooms.forEach(room => {
      const lastMessage = room.messages[0]?.content || '暂无消息';
      const truncatedMessage = lastMessage.length > 15 ? lastMessage.substring(0, 15) + '...' : lastMessage;
      console.log(`│ ${room.roomId.toString().padEnd(11)} │ ${room.roomName.padEnd(11)} │ ${room.creator.padEnd(11)} │ ${truncatedMessage.padEnd(11)} │`);
    });
    
    console.log('└─────────────┴─────────────┴─────────────┴─────────────┘');

    // 5. 统计信息
    console.log('\n5️⃣ 系统统计信息...');
    const userCount = await prisma.user.count();
    const roomCount = await prisma.room.count();
    const messageCount = await prisma.message.count();
    const adminCount = await prisma.user.count({
      where: { role: UserRole.ADMIN },
    });

    console.log(`👥 总用户数: ${userCount}`);
    console.log(`👑 管理员数: ${adminCount}`);
    console.log(`🏠 总房间数: ${roomCount}`);
    console.log(`💬 总消息数: ${messageCount}`);

    // 6. 测试管理员权限
    console.log('\n6️⃣ 测试管理员权限...');
    
    // 管理员可以删除任何房间
    const testRoom = await prisma.room.findFirst({
      where: { roomName: '技术讨论' },
    });

    if (testRoom) {
      console.log(`✅ 管理员可以访问房间: ${testRoom.roomName} (ID: ${testRoom.roomId})`);
    }

    // 管理员可以查看所有消息
    const allMessages = await prisma.message.findMany({
      take: 5,
      orderBy: { createdAt: 'desc' },
      include: {
        room: {
          select: { roomName: true },
        },
      },
    });

    console.log('\n📝 最近5条消息:');
    allMessages.forEach((msg, index) => {
      console.log(`   ${index + 1}. [${msg.room.roomName}] ${msg.sender}: ${msg.content}`);
    });

    console.log('\n✅ 管理员功能测试完成！');
    console.log('\n💡 使用说明:');
    console.log('1. 使用 admin/admin123 登录系统');
    console.log('2. 管理员可以删除任何房间和消息');
    console.log('3. 管理员可以查看所有用户信息');
    console.log('4. 普通用户只能管理自己创建的内容');

  } catch (error) {
    console.error('❌ 测试过程中出现错误:', error);
  } finally {
    await prisma.$disconnect();
  }
}

// 运行测试
testAdminFeatures();
