import { PrismaClient } from '@prisma/client';

// 全局变量声明，用于开发环境的热重载
declare global {
  var __prisma: PrismaClient | undefined;
}

// 创建 Prisma 客户端实例
export const prisma = globalThis.__prisma || new PrismaClient({
  log: process.env.NODE_ENV === 'development' ? ['query', 'error', 'warn'] : ['error'],
});

// 开发环境下保存实例到全局变量
if (process.env.NODE_ENV === 'development') {
  globalThis.__prisma = prisma;
}

// 数据库连接测试
export async function testDatabaseConnection() {
  try {
    await prisma.$connect();
    console.log('✅ 数据库连接成功');
    return true;
  } catch (error) {
    console.error('❌ 数据库连接失败:', error);
    return false;
  }
}

// 优雅关闭数据库连接
export async function closeDatabaseConnection() {
  try {
    await prisma.$disconnect();
    console.log('✅ 数据库连接已关闭');
  } catch (error) {
    console.error('❌ 关闭数据库连接失败:', error);
  }
}

// 数据库健康检查
export async function checkDatabaseHealth() {
  try {
    // 执行简单查询测试连接
    await prisma.$queryRaw`SELECT 1`;
    return {
      status: 'healthy',
      timestamp: new Date().toISOString(),
      message: '数据库连接正常'
    };
  } catch (error) {
    return {
      status: 'unhealthy',
      timestamp: new Date().toISOString(),
      message: '数据库连接异常',
      error: error instanceof Error ? error.message : '未知错误'
    };
  }
}
