'use client';

import React from 'react';
import { Avatar, Typography } from 'antd';
import { Message } from '../types/chat';
import styles from './MessageItem.module.css';

const { Text } = Typography;

interface MessageItemProps {
  message: Message;
  isOwn: boolean;
}

export default function MessageItem({ message, isOwn }: MessageItemProps) {
  const formatTime = (timestamp: number) => {
    const date = new Date(timestamp);
    const now = new Date();
    
    // 如果是今天，显示时间
    if (date.toDateString() === now.toDateString()) {
      return date.toLocaleTimeString('zh-CN', { 
        hour: '2-digit', 
        minute: '2-digit' 
      });
    }
    
    // 如果是昨天，显示"昨天"
    const yesterday = new Date(now);
    yesterday.setDate(yesterday.getDate() - 1);
    if (date.toDateString() === yesterday.toDateString()) {
      return '昨天 ' + date.toLocaleTimeString('zh-CN', { 
        hour: '2-digit', 
        minute: '2-digit' 
      });
    }
    
    // 其他显示日期和时间
    return date.toLocaleDateString('zh-CN', { 
      month: '2-digit', 
      day: '2-digit',
      hour: '2-digit', 
      minute: '2-digit' 
    });
  };

  return (
    <div className={`${styles.messageItem} ${isOwn ? styles.own : styles.other}`}>
      {!isOwn && (
        <Avatar 
          size="small" 
          className={styles.avatar}
          style={{ backgroundColor: '#1890ff' }}
        >
          {message.sender.charAt(0).toUpperCase()}
        </Avatar>
      )}
      
      <div className={styles.messageContent}>
        {!isOwn && (
          <div className={styles.senderInfo}>
            <div className={styles.senderName}>
              {message.sender}
            </div>
          </div>
        )}
        
        {isOwn && (
          <div className={styles.senderInfo}>
            <Avatar 
              size="small" 
              className={styles.avatar}
              style={{ backgroundColor: '#52c41a' }}
            >
              {message.sender.charAt(0).toUpperCase()}
            </Avatar>
            <div className={styles.senderName}>
              {message.sender}
            </div>
          </div>
        )}
        
        <div className={`${styles.messageBubble} ${isOwn ? styles.ownBubble : styles.otherBubble}`}>
          <Text className={styles.messageText}>
            {message.content}
          </Text>
        </div>
        
        <div className={`${styles.messageTime} ${isOwn ? styles.ownTime : styles.otherTime}`}>
          {formatTime(message.time)}
        </div>
      </div>
    </div>
  );
}
